# Guide du Canvas et Interactions Visuelles

> Documentation technique pour étendre le Canvas FoundryVTT v13 et ses objets visuels.

## Vue d'ensemble

Le **Canvas** est le coeur graphique de FoundryVTT. Il gère l'affichage des scènes, tokens, templates, notes et tous les éléments visuels de la table de jeu. Un système peut étendre ces composants pour personnaliser le rendu et le comportement des éléments sur la carte.

```
┌─────────────────────────────────────────────────────────────────────┐
│                     ARCHITECTURE DU CANVAS                          │
├─────────────────────────────────────────────────────────────────────┤
│                                                                     │
│  Canvas (PIXI.Application)                                          │
│  ├── BackgroundLayer        Fond de carte                           │
│  ├── DrawingsLayer          Dessins libres                          │
│  ├── GridLayer              Grille                                  │
│  ├── TemplateLayer          MeasuredTemplates (zones d'effet)       │
│  ├── TokenLayer             Tokens                                  │
│  ├── WallsLayer             Murs et portes                          │
│  ├── LightingLayer          Éclairage                               │
│  ├── SoundsLayer            Ambiances sonores                       │
│  ├── NotesLayer             Notes/Pins de journal                   │
│  └── ControlsLayer          UI de contrôle                          │
│                                                                     │
│  Chaque Layer contient des Placeables (objets visuels)              │
│  Chaque Placeable est lié à un Document (TokenDocument, etc.)       │
│                                                                     │
└─────────────────────────────────────────────────────────────────────┘
```

## Relation Document / Placeable

Dans FoundryVTT, les objets visuels suivent un pattern Document-View :

```
┌─────────────────┐         ┌─────────────────┐
│ TokenDocument   │◄────────│     Token       │
│ (données)       │         │ (rendu visuel)  │
│                 │         │                 │
│ - x, y          │         │ - PIXI.Graphics │
│ - rotation      │         │ - Animations    │
│ - texture       │         │ - Interactions  │
│ - actorId       │         │                 │
└─────────────────┘         └─────────────────┘
        │                           │
        │                           │
        ▼                           ▼
┌─────────────────┐         ┌─────────────────┐
│     Actor       │         │   TokenLayer    │
│ (acteur lié)    │         │ (conteneur)     │
└─────────────────┘         └─────────────────┘
```

Le **Document** (TokenDocument) stocke les données persistantes.
Le **Placeable** (Token) gère le rendu PIXI et les interactions utilisateur.

---

## Tokens

### Token5e - Classe Token personnalisée

La classe `Token5e` étend le token de base pour ajouter des comportements spécifiques au système :

```javascript
// dnd5e/module/canvas/token.mjs
export default class Token5e extends foundry.canvas.placeables.Token {

  /**
   * Flash l'anneau du token lorsqu'il est ciblé
   */
  static onTargetToken(user, token, targeted) {
    if ( !targeted ) return;
    if ( !token.hasDynamicRing ) return;
    const color = Color.from(user.color);
    token.ring.flashColor(color, { duration: 500, easing: token.ring.constructor.easeTwoPeaks });
  }

  /* -------------------------------------------- */

  /**
   * Calcul du chemin de mouvement avec blocage par tokens
   */
  findMovementPath(waypoints, options) {
    // Vérifier si le blocage de tokens est activé
    if ( (game.settings.get("dnd5e", "movementAutomation") !== "full") 
      || !this.document.actor?.system.isCreature
      || this.document.actor.statuses.intersects(CONFIG.DND5E.neverBlockStatuses) ) {
      return super.findMovementPath(waypoints, options);
    }

    // Obtenir tous les espaces de grille comme waypoints
    waypoints = this.document.getCompleteMovementPath(waypoints);

    // Filtrer les waypoints non bloquants
    const grid = this.document.parent.grid;
    waypoints = waypoints.filter((waypoint, i) => {
      return !waypoint.intermediate || 
        this.layer.isOccupiedGridSpaceBlocking(grid.getOffset(waypoints[i + 1]), this);
    });
    return super.findMovementPath(waypoints, options);
  }

  /* -------------------------------------------- */

  /**
   * Calcul du coût de mouvement (terrain difficile)
   */
  _getMovementCostFunction(options) {
    const costFunction = super._getMovementCostFunction(options);
    if ( game.settings.get("dnd5e", "movementAutomation") === "none" ) return costFunction;

    const ignoredDifficultTerrain = this.actor?.system.attributes?.movement?.ignoredDifficultTerrain ?? new Set();
    const ignoreDifficult = ["all", "nonmagical"].some(i => ignoredDifficultTerrain.has(i));

    return (from, to, distance, segment) => {
      const cost = costFunction(from, to, distance, segment);
      
      // Si le terrain est déjà difficile, pas de cumul
      if ( segment.terrain?.difficultTerrain ) return cost;
      
      // Si on ignore le terrain difficile, pas de surcoût
      if ( ignoreDifficult ) return cost;
      
      // Vérifier si l'espace est occupé (terrain difficile dû aux tokens)
      if ( !this.layer.isOccupiedGridSpaceDifficult(to, this) ) return cost;
      
      // Terrain difficile : double le coût
      return cost + distance;
    };
  }

  /* -------------------------------------------- */

  /**
   * Dessin personnalisé de la barre de PV
   */
  _drawBar(number, bar, data) {
    if ( data.attribute === "attributes.hp" ) return this._drawHPBar(number, bar, data);
    return super._drawBar(number, bar, data);
  }

  _drawHPBar(number, bar, data) {
    let {value, max, effectiveMax, temp, tempmax} = this.document.actor.system.attributes.hp;
    temp = Number(temp || 0);
    tempmax = Number(tempmax || 0);

    // Différencier max effectif et max affiché
    effectiveMax = Math.max(0, effectiveMax);
    let displayMax = max + (tempmax > 0 ? tempmax : 0);

    // Calculer les pourcentages
    const tempPct = Math.clamp(temp, 0, displayMax) / displayMax;
    const colorPct = Math.clamp(value, 0, effectiveMax) / displayMax;
    const hpColor = dnd5e.documents.Actor5e.getHPColor(value, effectiveMax);

    // Couleurs
    const blk = 0x000000;
    const c = CONFIG.DND5E.tokenHPColors;

    // Dimensions
    let s = canvas.dimensions.uiScale;
    const bw = this.w;
    const bh = 8 * (this.document.height >= 2 ? 1.5 : 1) * s;

    // Dessiner la barre
    bar.clear();
    bar.beginFill(blk, 0.5).lineStyle(s, blk, 1.0).drawRoundedRect(0, 0, bw, bh, 3 * s);

    // PV temporaires max (bonus)
    if (tempmax > 0) {
      const pct = max / effectiveMax;
      bar.beginFill(c.tempmax, 1.0).drawRoundedRect(pct * bw, 0, (1 - pct) * bw, bh, 2 * s);
    }
    // Malus de PV max
    else if (tempmax < 0) {
      const pct = (max + tempmax) / max;
      bar.beginFill(c.negmax, 1.0).drawRoundedRect(pct * bw, 0, (1 - pct) * bw, bh, 2 * s);
    }

    // Barre de santé principale
    bar.beginFill(hpColor, 1.0).drawRoundedRect(0, 0, colorPct * bw, bh, 2 * s);

    // PV temporaires
    if ( temp > 0 ) {
      bar.beginFill(c.temp, 1.0).drawRoundedRect(s, s, (tempPct * bw) - (2 * s), bh - (2 * s), s);
    }

    bar.position.set(0, (number === 0) ? (this.h - bh) : 0);
  }

  /* -------------------------------------------- */

  /**
   * Personnalisation des couleurs de l'anneau dynamique
   */
  getRingColors() {
    return this.document.getRingColors();
  }

  getRingEffects() {
    return this.document.getRingEffects();
  }
}
```

### Enregistrement de la classe Token

```javascript
// Dans le hook init
Hooks.once("init", () => {
  CONFIG.Token.objectClass = Token5e;
});
```

### Création de Token personnalisé

```javascript
// Créer un nouveau token
export default class MonToken extends foundry.canvas.placeables.Token {

  /** @override */
  async _draw() {
    await super._draw();
    // Ajouter des éléments graphiques personnalisés
    this.customOverlay = this.addChild(new PIXI.Graphics());
    this._drawCustomOverlay();
  }

  _drawCustomOverlay() {
    if ( !this.customOverlay ) return;
    this.customOverlay.clear();
    
    // Dessiner un indicateur personnalisé
    if ( this.actor?.getFlag("mon-systeme", "marked") ) {
      this.customOverlay
        .lineStyle(3, 0xff0000, 0.8)
        .drawCircle(this.w / 2, this.h / 2, Math.max(this.w, this.h) / 2 + 5);
    }
  }

  /** @override */
  _refresh(options) {
    super._refresh(options);
    this._drawCustomOverlay();
  }

  /** @override */
  _destroy(options) {
    this.customOverlay?.destroy();
    super._destroy(options);
  }
}
```

---

## TokenLayer personnalisé

La classe `TokenLayer5e` gère les interactions entre tokens :

```javascript
// dnd5e/module/canvas/layers/tokens.mjs
export default class TokenLayer5e extends foundry.canvas.layers.TokenLayer {

  /**
   * Vérifie si un espace de grille bloque le mouvement
   * @param {GridOffset3D} gridSpace - L'espace à vérifier
   * @param {Token5e} token - Le token qui se déplace
   * @param {object} [options] - Options additionnelles
   * @returns {boolean}
   */
  isOccupiedGridSpaceBlocking(gridSpace, token, { preview=false }={}) {
    const tokenSize = CONFIG.DND5E.actorSizes[token.actor?.system.traits.size]?.numerical ?? 2;
    const modernRules = game.settings.get("dnd5e", "rulesVersion") === "modern";
    const halflingNimbleness = token.actor?.getFlag("dnd5e", "halflingNimbleness");

    const found = this.#getRelevantOccupyingTokens(gridSpace, token, { preview }).filter(t => {
      // Seules les créatures bloquent
      if ( !t.actor?.system.isCreature ) return false;

      // Les tokens amicaux ne bloquent pas
      if ( token.document.disposition === t.document.disposition ) return false;

      // Statuts qui ne bloquent jamais (inconscient, mort, etc.)
      if ( t.actor.statuses.intersects(CONFIG.DND5E.neverBlockStatuses) ) return false;

      const occupiedSize = CONFIG.DND5E.actorSizes[t.actor?.system.traits.size]?.numerical ?? 2;

      // Règles modernes : les créatures Tiny ne bloquent pas
      if ( modernRules && (occupiedSize === 0) ) return false;

      // Halfling Nimbleness : les grandes créatures ne bloquent pas
      if ( halflingNimbleness && (occupiedSize > tokenSize) ) return false;

      // Différence de taille < 2 = blocage
      return Math.abs(tokenSize - occupiedSize) < 2;
    });

    // Hook pour personnalisation
    Hooks.callAll("dnd5e.determineOccupiedGridSpaceBlocking", gridSpace, token, { preview }, found);
    return found.size > 0;
  }

  /**
   * Vérifie si un espace de grille cause du terrain difficile
   */
  isOccupiedGridSpaceDifficult(gridSpace, token, { preview=false }={}) {
    const modernRules = game.settings.get("dnd5e", "rulesVersion") === "modern";

    const found = this.#getRelevantOccupyingTokens(gridSpace, token, { preview }).filter(t => {
      if ( !t.actor?.system.isCreature ) return false;

      const friendlyToken = token.document.disposition === t.document.disposition;
      
      // Règles modernes : les alliés ne sont pas terrain difficile
      if ( modernRules && friendlyToken ) return false;

      const occupiedSize = CONFIG.DND5E.actorSizes[t.actor?.system.traits.size]?.numerical ?? 2;
      
      // Règles modernes : Tiny n'est pas terrain difficile
      if ( modernRules && (occupiedSize === 0) ) return false;

      return true;
    });

    Hooks.callAll("dnd5e.determineOccupiedGridSpaceDifficult", gridSpace, token, { preview }, found);
    return found.size > 0;
  }

  /**
   * Récupère les tokens occupant un espace de grille
   */
  #getRelevantOccupyingTokens(gridSpace, token, { preview=false }={}) {
    const grid = canvas.grid;
    if ( grid.isGridless ) return [];
    
    const topLeft = grid.getTopLeftPoint(gridSpace);
    const rect = new PIXI.Rectangle(topLeft.x, topLeft.y, grid.sizeX, grid.sizeY);
    const lowerElevation = gridSpace.k * grid.distance;
    const upperElevation = (gridSpace.k + 1) * grid.distance;

    return game.canvas.tokens.quadtree.getObjects(rect, {
      collisionTest: ({ t }) => {
        if ( t === token ) return false;
        if ( canvas.tokens.controlled.includes(t) ) return false;
        if ( t.document.hidden ) return false;
        if ( preview && !t.visible ) return false;
        if ( t.document.disposition === CONST.TOKEN_DISPOSITIONS.SECRET ) return false;

        const occupiedElevation = t.document._source.elevation;
        if ( (occupiedElevation < lowerElevation) || (occupiedElevation >= upperElevation) ) return false;

        const gridSpaces = t.document.getOccupiedGridSpaceOffsets(t.document._source);
        return gridSpaces.some(coord => (coord.i === gridSpace.i) && (coord.j === gridSpace.j));
      }
    });
  }
}
```

### Enregistrement du TokenLayer

```javascript
Hooks.once("init", () => {
  CONFIG.Canvas.layers.tokens.layerClass = TokenLayer5e;
});
```

---

## Measured Templates (Zones d'effet)

### AbilityTemplate - Templates pour sorts et capacités

La classe `AbilityTemplate` gère le placement interactif des zones d'effet :

```javascript
// dnd5e/module/canvas/ability-template.mjs
export default class AbilityTemplate extends foundry.canvas.placeables.MeasuredTemplate {

  /**
   * Crée des templates depuis une Activity
   * @param {Activity} activity - L'activité source
   * @param {object} [options={}] - Options de création
   * @returns {AbilityTemplate[]|null}
   */
  static fromActivity(activity, options={}) {
    const target = activity.target?.template ?? {};
    const templateShape = dnd5e.config.areaTargetTypes[target.type]?.template;
    if ( !templateShape ) return null;

    // Préparer les données du template
    const rollData = activity.getRollData();
    const templateData = foundry.utils.mergeObject({
      t: templateShape,
      user: game.user.id,
      distance: target.size,
      direction: 0,
      x: 0,
      y: 0,
      fillColor: game.user.color,
      flags: { dnd5e: {
        dimensions: {
          size: target.size,
          width: target.width,
          height: target.height,
          adjustedSize: target.type === "radius"
        },
        item: activity.item.uuid,
        origin: activity.uuid,
        spellLevel: rollData.item.level
      } }
    }, options);

    // Configuration spécifique selon le type
    switch ( templateShape ) {
      case "cone":
        templateData.angle = CONFIG.MeasuredTemplate.defaults.angle;
        break;
      case "rect":
        templateData.width = target.size;
        if ( game.settings.get("dnd5e", "gridAlignedSquareTemplates") ) {
          templateData.distance = Math.hypot(target.size, target.size);
          templateData.direction = 45;
        } else {
          templateData.t = "ray";  // Permet la rotation
        }
        break;
      case "ray":
        templateData.width = target.width ?? canvas.dimensions.distance;
        break;
    }

    // Hook de pré-création
    if ( Hooks.call("dnd5e.preCreateActivityTemplate", activity, templateData) === false ) return null;

    // Créer les templates
    const cls = CONFIG.MeasuredTemplate.documentClass;
    const created = Array.fromRange(target.count || 1).map(() => {
      const template = new cls(foundry.utils.deepClone(templateData), { parent: canvas.scene });
      const object = new this(template);
      object.activity = activity;
      object.item = activity.item;
      object.actorSheet = activity.actor?.sheet || null;
      return object;
    });

    Hooks.callAll("dnd5e.createActivityTemplate", activity, created);
    return created;
  }

  /* -------------------------------------------- */

  /**
   * Affiche la prévisualisation et attend le placement
   * @returns {Promise} Résout avec le template créé
   */
  drawPreview() {
    const initialLayer = canvas.activeLayer;

    // Dessiner le template
    this.draw();
    this.layer.activate();
    this.layer.preview.addChild(this);

    // Minimiser la feuille d'origine
    this.actorSheet?.minimize();

    return this.activatePreviewListeners(initialLayer);
  }

  /**
   * Active les listeners pour le placement interactif
   */
  activatePreviewListeners(initialLayer) {
    return new Promise((resolve, reject) => {
      this.#initialLayer = initialLayer;
      this.#events = {
        cancel: this._onCancelPlacement.bind(this),
        confirm: this._onConfirmPlacement.bind(this),
        move: this._onMovePlacement.bind(this),
        resolve,
        reject,
        rotate: this._onRotatePlacement.bind(this)
      };

      // Activer les listeners
      canvas.stage.on("mousemove", this.#events.move);
      canvas.stage.on("mouseup", this.#events.confirm);
      canvas.app.view.oncontextmenu = this.#events.cancel;
      canvas.app.view.onwheel = this.#events.rotate;
    });
  }

  /* -------------------------------------------- */

  /**
   * Déplacement du template avec la souris
   */
  _onMovePlacement(event) {
    event.stopPropagation();
    const now = Date.now();
    if ( now - this.#moveTime <= 20 ) return;  // Throttle 20ms
    
    const center = event.data.getLocalPosition(this.layer);
    const updates = this.getSnappedPosition(center);

    // Ajuster la taille si un token est survolé (pour les radius)
    const baseDistance = this.document.flags.dnd5e?.dimensions?.size;
    if ( this.document.flags.dnd5e?.dimensions?.adjustedSize && baseDistance ) {
      const rectangle = new PIXI.Rectangle(center.x, center.y, 1, 1);
      const hoveredToken = canvas.tokens.quadtree.getObjects(rectangle, {
        collisionTest: ({ t }) => t.visible && !t.document.isSecret
      }).first();
      
      if ( hoveredToken && (hoveredToken !== this.#hoveredToken) ) {
        this.#hoveredToken = hoveredToken;
        this.#hoveredToken._onHoverIn(event);
        const size = Math.max(hoveredToken.document.width, hoveredToken.document.height);
        updates.distance = baseDistance + (size * canvas.grid.distance / 2);
      } else if ( !hoveredToken && this.#hoveredToken ) {
        this.#hoveredToken._onHoverOut(event);
        this.#hoveredToken = null;
        updates.distance = baseDistance;
      }
    }

    this.document.updateSource(updates);
    this.refresh();
    this.#moveTime = now;
  }

  /**
   * Rotation du template avec la molette
   */
  _onRotatePlacement(event) {
    if ( this.document.t === "rect" ) return;
    if ( event.ctrlKey ) event.preventDefault();
    event.stopPropagation();
    
    const delta = canvas.grid.type > CONST.GRID_TYPES.SQUARE ? 30 : 15;
    const snap = event.shiftKey ? delta : 5;
    const update = {direction: this.document.direction + (snap * Math.sign(event.deltaY))};
    this.document.updateSource(update);
    this.refresh();
  }

  /**
   * Confirmation du placement (clic gauche)
   */
  async _onConfirmPlacement(event) {
    await this._finishPlacement(event);
    const destination = canvas.templates.getSnappedPoint({ x: this.document.x, y: this.document.y });
    this.document.updateSource(destination);
    this.#events.resolve(canvas.scene.createEmbeddedDocuments("MeasuredTemplate", [this.document.toObject()]));
  }

  /**
   * Annulation du placement (clic droit)
   */
  async _onCancelPlacement(event) {
    await this._finishPlacement(event);
    this.#events.reject();
  }
}
```

### Utilisation des templates

```javascript
// Placer un template pour un sort
async function placeSpellTemplate(activity) {
  const templates = AbilityTemplate.fromActivity(activity);
  if ( !templates ) return null;

  try {
    const placed = [];
    for ( const template of templates ) {
      const result = await template.drawPreview();
      if ( result ) placed.push(...result);
    }
    return placed;
  } catch (error) {
    // L'utilisateur a annulé le placement
    return null;
  }
}

// Créer un template manuellement
async function createCustomTemplate() {
  const templateData = {
    t: "circle",
    x: 500,
    y: 500,
    distance: 20,  // 20 pieds de rayon
    direction: 0,
    fillColor: "#ff0000",
    flags: {
      "mon-systeme": {
        effect: "explosion"
      }
    }
  };

  return canvas.scene.createEmbeddedDocuments("MeasuredTemplate", [templateData]);
}
```

---

## Detection Modes

Les Detection Modes définissent comment un token peut percevoir d'autres tokens :

```javascript
// dnd5e/module/canvas/detection-modes/blindsight.mjs
export class DetectionModeBlindsight extends foundry.canvas.perception.DetectionMode {
  constructor() {
    super({
      id: "blindsight",
      label: "DND5E.SenseBlindsight",
      type: DetectionMode.DETECTION_TYPES.OTHER,
      walls: true,   // Bloqué par les murs
      angle: false   // Vision à 360°
    });
  }

  /**
   * Filtre visuel appliqué aux tokens détectés
   */
  static getDetectionFilter() {
    return this._detectionFilter ??= OutlineOverlayFilter.create({
      outlineColor: [1, 1, 1, 1],
      knockout: true,
      wave: true
    });
  }

  /**
   * Détermine si la source peut détecter la cible
   */
  _canDetect(visionSource, target) {
    // Ne détecte pas si la source est enterrée
    if ( visionSource.object.document.hasStatusEffect(CONFIG.specialStatusEffects.BURROW) ) return false;
    
    // Ne détecte pas les cibles enterrées
    if ( target instanceof foundry.canvas.placeables.Token ) {
      if ( target.document.hasStatusEffect(CONFIG.specialStatusEffects.BURROW) ) return false;
    }
    return true;
  }

  /**
   * Test de ligne de vue
   */
  _testLOS(visionSource, mode, target, test) {
    return !CONFIG.Canvas.polygonBackends.sight.testCollision(
      { x: visionSource.x, y: visionSource.y },
      test.point,
      {
        type: "sight",
        mode: "any",
        source: visionSource,
        // La vision aveugle est bloquée par la couverture totale
        // mais pas par les fenêtres (useThreshold: false)
        useThreshold: false
      }
    );
  }
}

// Enregistrement du mode
CONFIG.Canvas.detectionModes.blindsight = new DetectionModeBlindsight();
```

### Créer un mode de détection personnalisé

```javascript
export class DetectionModeTremorsense extends foundry.canvas.perception.DetectionMode {
  constructor() {
    super({
      id: "tremorsense",
      label: "MON_SYSTEME.SenseTremorsense",
      type: DetectionMode.DETECTION_TYPES.OTHER,
      walls: false,  // Non bloqué par les murs
      angle: false
    });
  }

  static getDetectionFilter() {
    return this._detectionFilter ??= OutlineOverlayFilter.create({
      outlineColor: [0.8, 0.6, 0.2, 1],  // Couleur terre
      knockout: true
    });
  }

  _canDetect(visionSource, target) {
    // Ne fonctionne que sur les cibles au sol (même élévation)
    if ( target instanceof foundry.canvas.placeables.Token ) {
      const sourceElevation = visionSource.object.document.elevation;
      const targetElevation = target.document.elevation;
      if ( Math.abs(sourceElevation - targetElevation) > 5 ) return false;
      
      // Ne détecte pas les créatures volantes
      if ( target.document.hasStatusEffect("fly") ) return false;
    }
    return true;
  }

  _testLOS(visionSource, mode, target, test) {
    // Pas de test LOS - détecte à travers les murs
    return true;
  }
}

// Enregistrement
Hooks.once("init", () => {
  CONFIG.Canvas.detectionModes.tremorsense = new DetectionModeTremorsense();
});
```

---

## Token Placement

La classe `TokenPlacement` gère le placement interactif de tokens :

```javascript
// dnd5e/module/canvas/token-placement.mjs
export default class TokenPlacement {
  constructor(config) {
    this.config = config;
  }

  /**
   * Configuration des placements
   * @type {TokenPlacementConfiguration}
   */
  config;

  /**
   * Méthode statique pour placer des tokens
   * @param {TokenPlacementConfiguration} config
   * @returns {Promise<TokenPlacementData[]>}
   */
  static place(config) {
    const placement = new this(config);
    return placement.place();
  }

  /**
   * Effectue le placement
   */
  async place() {
    this.#createPreviews();
    try {
      const placements = [];
      let total = 0;
      const uniqueTokens = new Map();
      
      while ( this.#currentPlacement < this.config.tokens.length - 1 ) {
        this.#currentPlacement++;
        
        // Ajouter la prévisualisation
        const obj = canvas.tokens.preview.addChild(this.#previews[this.#currentPlacement].object);
        await obj.draw();
        obj.eventMode = "none";
        
        // Attendre le placement utilisateur
        const placement = await this.#requestPlacement();
        if ( placement ) {
          const actorId = placement.prototypeToken.parent.id;
          uniqueTokens.set(actorId, (uniqueTokens.get(actorId) ?? -1) + 1);
          placement.index = { total: total++, unique: uniqueTokens.get(actorId) };
          placements.push(placement);
        } else {
          obj.clear();
        }
      }
      return placements;
    } finally {
      this.#destroyPreviews();
    }
  }

  /**
   * Crée les prévisualisations de tokens
   */
  #createPreviews() {
    this.#placements = [];
    this.#previews = [];
    
    for ( const prototypeToken of this.config.tokens ) {
      const tokenData = prototypeToken.toObject();
      tokenData.sight.enabled = false;
      tokenData._id = foundry.utils.randomID();
      if ( tokenData.randomImg ) tokenData.texture.src = prototypeToken.actor.img;
      
      const cls = getDocumentClass("Token");
      const doc = new cls(tokenData, { parent: canvas.scene });
      
      this.#placements.push({
        prototypeToken,
        x: 0,
        y: 0,
        elevation: this.config.origin?.elevation ?? 0,
        rotation: tokenData.rotation ?? 0
      });
      this.#previews.push(doc);
    }
  }

  /**
   * Active les listeners pour le placement
   */
  #requestPlacement() {
    return new Promise((resolve, reject) => {
      this.#events = {
        confirm: this.#onConfirmPlacement.bind(this),
        move: this.#onMovePlacement.bind(this),
        resolve,
        reject,
        rotate: this.#onRotatePlacement.bind(this),
        skip: this.#onSkipPlacement.bind(this)
      };

      canvas.stage.on("mousemove", this.#events.move);
      canvas.stage.on("mousedown", this.#events.confirm);
      canvas.app.view.oncontextmenu = this.#events.skip;
      canvas.app.view.onwheel = this.#events.rotate;
    });
  }

  /**
   * Ajuste le numéro d'un token unlinked
   */
  static adjustAppendedNumber(tokenDocument, placement) {
    const regex = new RegExp(/\((\d+)\)$/);
    const match = tokenDocument.name?.match(regex);
    if ( !match ) return;
    
    const name = tokenDocument.name.replace(regex, `(${Number(match[1]) + placement.index.unique})`);
    if ( tokenDocument instanceof TokenDocument ) tokenDocument.updateSource({ name });
    else tokenDocument.name = name;
  }
}
```

### Utilisation du placement

```javascript
// Placer des tokens d'invocation
async function placeConjuredCreatures(actor, conjuredActors) {
  const tokens = conjuredActors.map(a => a.prototypeToken);
  
  const placements = await TokenPlacement.place({
    origin: actor.getActiveTokens()[0]?.document,
    tokens
  });

  // Créer les tokens sur la scène
  const tokenDataArray = placements.map(p => {
    const data = p.prototypeToken.toObject();
    data.x = p.x;
    data.y = p.y;
    data.elevation = p.elevation;
    data.rotation = p.rotation;
    TokenPlacement.adjustAppendedNumber(data, p);
    return data;
  });

  return canvas.scene.createEmbeddedDocuments("Token", tokenDataArray);
}
```

---

## Token Ruler

La règle de mouvement personnalisée avec indication de vitesse :

```javascript
// dnd5e/module/canvas/ruler.mjs
export default class TokenRuler5e extends foundry.canvas.placeables.tokens.TokenRuler {

  /**
   * Style des waypoints basé sur la vitesse
   */
  _getWaypointStyle(waypoint) {
    if ( !waypoint.explicit && waypoint.next && waypoint.previous && waypoint.actionConfig.visualize
      && waypoint.next.actionConfig.visualize && (waypoint.action === waypoint.next.action)
      && (waypoint.unreachable || !waypoint.next.unreachable) ) return { radius: 0 };
    
    const user = game.users.get(waypoint.userId);
    const scale = canvas.dimensions.uiScale;
    const style = {
      radius: 6 * scale,
      color: user?.color ?? 0x000000,
      alpha: waypoint.explicit ? 1 : 0.5
    };
    return this.#getSpeedBasedStyle(waypoint, style);
  }

  /**
   * Style des segments basé sur la vitesse
   */
  _getSegmentStyle(waypoint) {
    const style = super._getSegmentStyle(waypoint);
    return this.#getSpeedBasedStyle(waypoint, style);
  }

  /**
   * Applique les couleurs selon la distance parcourue vs vitesse
   */
  #getSpeedBasedStyle(waypoint, style) {
    // Si ce n'est pas le mouvement de l'utilisateur courant, style par défaut
    if ( !(game.user.id in this.token._plannedMovement)
      || CONFIG.Token.movement.actions[waypoint.action]?.teleport ) return style;

    // Récupérer la vitesse de l'acteur
    const movement = this.token.actor?.system.attributes?.movement;
    if ( !movement || !this.token.actor?.system.isCreature ) return style;
    
    let currActionSpeed = movement[waypoint.action] ?? 0;

    // Fallback vers walk si applicable
    if ( CONFIG.DND5E.movementTypes[waypoint.action]?.walkFallback
      || !CONFIG.DND5E.movementTypes[waypoint.action] ) {
      currActionSpeed = Math.max(currActionSpeed, movement.walk);
    }

    // Couleurs : normal si <= vitesse, double si <= 2x, triple sinon
    const { normal, double, triple } = CONFIG.DND5E.tokenRulerColors;
    const increment = (waypoint.measurement.cost - .1) / currActionSpeed;
    
    if ( increment <= 1 ) style.color = normal ?? style.color;
    else if ( increment <= 2 ) style.color = double ?? style.color;
    else style.color = triple ?? style.color;
    
    return style;
  }
}
```

---

## Map Location Control Icon

Icône personnalisée pour les notes de type "Map Location" :

```javascript
// dnd5e/module/canvas/map-location-control-icon.mjs
export default class MapLocationControlIcon extends PIXI.Container {
  constructor({code, size=40, ...style}={}, ...args) {
    super(...args);

    this.code = code;      // Texte à afficher (ex: "A1")
    this.size = size;
    this.style = style;

    this.renderMarker();
    this.refresh();
  }

  /**
   * Rendu du marqueur
   */
  renderMarker() {
    this.radius = this.size / 2;
    this.circle = [this.radius, this.radius, this.radius + 8];
    this.backgroundColor = this.style.backgroundColor;
    this.borderColor = this.style.borderHoverColor;

    // Zone interactive
    this.eventMode = "static";
    this.interactiveChildren = false;
    this.hitArea = new PIXI.Circle(...this.circle);
    this.cursor = "pointer";

    // Ombre portée
    this.shadow = this.addChild(new PIXI.Graphics());
    this.shadow.clear()
      .beginFill(this.style.shadowColor, 0.65)
      .drawCircle(this.radius + 8, this.radius + 8, this.radius + 10)
      .endFill();
    this.shadow.filters = [new PIXI.filters.BlurFilter(16)];

    // Effet 3D
    this.extrude = this.addChild(new PIXI.Graphics());
    this.extrude.clear()
      .beginFill(this.style.borderColor, 1.0)
      .drawCircle(this.radius + 2, this.radius + 2, this.radius + 9)
      .endFill();

    // Fond
    this.bg = this.addChild(new PIXI.Graphics());
    this.bg.clear()
      .beginFill(this.backgroundColor, 1.0)
      .lineStyle(2, this.style.borderColor, 1.0)
      .drawCircle(...this.circle)
      .endFill();

    // Texte du code
    this.text = new foundry.canvas.containers.PreciseText(this.code, this._getTextStyle(this.code.length, this.size));
    this.text.anchor.set(0.5, 0.5);
    this.text.position.set(this.radius, this.radius);
    this.addChild(this.text);

    // Bordure (pour le hover)
    this.border = this.addChild(new PIXI.Graphics());
    this.border.visible = false;
  }

  /**
   * Rafraîchit le rendu
   */
  refresh({ visible, iconColor, borderColor, borderVisible }={}) {
    if ( borderColor ) this.borderColor = borderColor;
    this.border.clear().lineStyle(2, this.borderColor, 1.0).drawCircle(...this.circle).endFill();
    if ( borderVisible !== undefined ) this.border.visible = borderVisible;
    if ( visible !== undefined ) this.visible = visible;
    return this;
  }

  /**
   * Style de texte adaptatif
   */
  _getTextStyle(characterCount, size) {
    const style = CONFIG.canvasTextStyle.clone();
    style.dropShadow = false;
    style.fill = Color.from(this.style.textColor);
    style.strokeThickness = 0;
    style.fontFamily = ["Signika"];
    if ( this.style.fontFamily ) style.fontFamily.unshift(this.style.fontFamily);
    style.fontSize = characterCount > 2 ? size * .5 : size * .6;
    return style;
  }
}
```

---

## Note personnalisée

```javascript
// dnd5e/module/canvas/note.mjs
export default class Note5e extends foundry.canvas.placeables.Note {
  
  /**
   * Permet aux pages de journal d'avoir des icônes personnalisées
   */
  _drawControlIcon() {
    const tint = Color.from(this.document.texture.tint || null);
    
    // Demander au système de page s'il a une icône personnalisée
    const systemIcon = this.page?.system?.getControlIcon?.({ size: this.document.iconSize, tint });
    if ( !systemIcon ) return super._drawControlIcon();
    
    // Positionner l'icône
    systemIcon.x -= (this.document.iconSize / 2);
    systemIcon.y -= (this.document.iconSize / 2);
    return systemIcon;
  }
}
```

---

## Enregistrement de tous les composants Canvas

```javascript
// Dans le hook init
Hooks.once("init", () => {
  // Token personnalisé
  CONFIG.Token.objectClass = canvas.Token5e;
  
  // TokenLayer personnalisé
  CONFIG.Canvas.layers.tokens.layerClass = canvas.TokenLayer5e;
  
  // Template personnalisé
  CONFIG.MeasuredTemplate.objectClass = canvas.AbilityTemplate;
  
  // Note personnalisée
  CONFIG.Note.objectClass = canvas.Note5e;
  
  // Ruler personnalisé
  CONFIG.Token.rulerClass = canvas.TokenRuler5e;
  
  // Modes de détection
  CONFIG.Canvas.detectionModes.blindsight = new canvas.detectionModes.DetectionModeBlindsight();
});
```

---

## Bonnes Pratiques

1. **Hériter des classes Foundry** - Toujours étendre les classes de base plutôt que de les remplacer.

2. **Utiliser les hooks** - Émettre des hooks pour permettre aux modules de personnaliser le comportement.

3. **Throttling** - Pour les événements fréquents (mousemove), implémenter un throttling pour les performances.

4. **Nettoyage** - Toujours implémenter `_destroy()` pour nettoyer les ressources PIXI.

5. **Prévisualisations** - Utiliser `canvas.tokens.preview` et `canvas.templates.preview` pour les objets temporaires.

6. **Configuration dans CONFIG** - Enregistrer les classes via CONFIG pour permettre leur remplacement.

---

*Documentation précédente : [07-templates.md](./07-templates.md) - Templates Handlebars*
