# Templates Handlebars

> Documentation technique pour FoundryVTT v13

## Vue d'ensemble

Les **Templates Handlebars** sont le moteur de rendu HTML de FoundryVTT. Ils permettent de créer des interfaces dynamiques en combinant HTML statique avec des expressions et helpers pour injecter des données.

Foundry utilise Handlebars avec des extensions personnalisées (helpers) qui facilitent la création d'interfaces de jeu.

```
┌─────────────────────────────────────────────────────────────────┐
│                 FLUX DE RENDU HANDLEBARS                        │
├─────────────────────────────────────────────────────────────────┤
│                                                                 │
│   ApplicationV2                                                 │
│       │                                                         │
│       ├── _prepareContext()  ──▶  { actor, items, ... }        │
│       │                                                         │
│       └── _renderHTML()                                         │
│               │                                                 │
│               ▼                                                 │
│       ┌───────────────┐                                         │
│       │   Template    │  template.hbs                           │
│       │   Handlebars  │  + context data                         │
│       └───────┬───────┘                                         │
│               │                                                 │
│               ▼                                                 │
│       ┌───────────────┐                                         │
│       │  HTML rendu   │  Injecté dans le DOM                    │
│       └───────────────┘                                         │
│                                                                 │
└─────────────────────────────────────────────────────────────────┘
```

## Syntaxe de Base

### Expressions

```handlebars
{{! Commentaire Handlebars - non rendu }}

{{! Variable simple }}
<h1>{{actor.name}}</h1>

{{! Accès aux propriétés imbriquées }}
<span>{{actor.system.attributes.hp.value}}</span>

{{! Expression avec helper }}
<span>{{localize "DND5E.HitPoints"}}</span>
```

### Échappement HTML

```handlebars
{{! Échappé (sécurisé) - convertit < > en entités }}
<p>{{description}}</p>

{{! Non échappé (HTML brut) - ATTENTION aux injections }}
<div>{{{richDescription}}}</div>
```

## Conditions

### {{#if}} / {{else}}

```handlebars
{{#if actor.system.attributes.hp.value}}
  <span class="hp">{{actor.system.attributes.hp.value}} HP</span>
{{else}}
  <span class="dead">Mort</span>
{{/if}}

{{! Condition inversée }}
{{#unless isEditable}}
  <span class="locked">🔒 Lecture seule</span>
{{/unless}}
```

### Opérateurs de comparaison (Foundry)

Foundry ajoute des helpers de comparaison :

```handlebars
{{! Égalité }}
{{#if (eq type "character")}}
  <div class="character-sheet">...</div>
{{/if}}

{{! Différence }}
{{#if (ne status "dead")}}
  <button>Attaquer</button>
{{/if}}

{{! Comparaisons numériques }}
{{#if (gt hp 0)}}Vivant{{/if}}
{{#if (gte level 5)}}Multi-attaque disponible{{/if}}
{{#if (lt hp maxHp)}}Blessé{{/if}}
{{#if (lte charges 0)}}Épuisé{{/if}}

{{! Logique booléenne }}
{{#if (and isOwner isEditable)}}
  <input type="text" value="{{name}}">
{{/if}}

{{#if (or isGM isOwner)}}
  <button>Modifier</button>
{{/if}}

{{#if (not isLocked)}}
  <button>Éditer</button>
{{/if}}
```

## Boucles

### {{#each}}

```handlebars
{{! Boucle sur un tableau }}
<ul class="items">
  {{#each items}}
    <li data-item-id="{{this.id}}">
      <img src="{{this.img}}" alt="{{this.name}}">
      <span>{{this.name}}</span>
    </li>
  {{/each}}
</ul>

{{! Avec index }}
{{#each skills}}
  <div class="skill" data-index="{{@index}}">
    {{@key}}: {{this.value}}
  </div>
{{/each}}

{{! Boucle sur un objet }}
{{#each actor.system.abilities}}
  <div class="ability">
    <label>{{@key}}</label>
    <span>{{this.value}}</span>
    <span>({{this.mod}})</span>
  </div>
{{/each}}
```

### Variables spéciales dans les boucles

```handlebars
{{#each items}}
  {{@index}}    {{! Index numérique (0, 1, 2...) }}
  {{@key}}      {{! Clé de l'objet }}
  {{@first}}    {{! true si premier élément }}
  {{@last}}     {{! true si dernier élément }}
  {{this}}      {{! Élément courant }}
  {{../parent}} {{! Accès au contexte parent }}
{{/each}}
```

## Helpers Foundry Natifs

### Localisation

```handlebars
{{! Traduction simple }}
<label>{{localize "DND5E.Strength"}}</label>

{{! Traduction avec interpolation }}
<p>{{localize "DND5E.LevelCount" level=5}}</p>

{{! Vérifier si une clé existe }}
{{#if (localize "MYSYSTEM.CustomKey")}}
  {{localize "MYSYSTEM.CustomKey"}}
{{/if}}
```

### Formulaires et Inputs

```handlebars
{{! Champ texte lié aux données }}
<input type="text" name="name" value="{{actor.name}}">

{{! Champ numérique }}
<input type="number" name="system.attributes.hp.value" 
       value="{{actor.system.attributes.hp.value}}" 
       min="0" max="{{actor.system.attributes.hp.max}}">

{{! Checkbox }}
<input type="checkbox" name="system.traits.inspiration" 
       {{checked actor.system.traits.inspiration}}>

{{! Select avec options }}
<select name="system.details.alignment">
  {{selectOptions alignments selected=actor.system.details.alignment}}
</select>

{{! Select avec localization }}
<select name="system.details.size">
  {{selectOptions config.sizes selected=actor.system.details.size localize=true}}
</select>
```

### Helpers de formatage

```handlebars
{{! Nombre signé (+/-) }}
<span>{{numberFormat modifier sign=true}}</span>
{{! Affiche: +3 ou -2 }}

{{! Formatage de nombre }}
<span>{{numberFormat gold decimals=2}}</span>

{{! Éditeur de texte riche }}
{{editor content=actor.system.details.biography 
         target="system.details.biography" 
         button=true 
         editable=isEditable}}

{{! FilePicker pour images }}
{{filePicker type="image" value=actor.img target="img"}}
```

## Partials

Les **partials** sont des templates réutilisables.

### Enregistrement d'un partial

```javascript
// Dans le hook init
Hooks.once("init", async () => {
  // Charger et enregistrer un partial
  await loadTemplates([
    "systems/mon-systeme/templates/partials/item-card.hbs",
    "systems/mon-systeme/templates/partials/ability-score.hbs"
  ]);
});
```

### Utilisation d'un partial

```handlebars
{{! Appel simple }}
{{> "systems/mon-systeme/templates/partials/item-card.hbs"}}

{{! Avec contexte explicite }}
{{> "systems/mon-systeme/templates/partials/ability-score.hbs" 
    ability=actor.system.abilities.str 
    label="Force"}}

{{! Dans une boucle }}
{{#each items}}
  {{> "systems/mon-systeme/templates/partials/item-card.hbs" item=this}}
{{/each}}
```

### Exemple de partial

```handlebars
{{! partials/ability-score.hbs }}
<div class="ability-score" data-ability="{{@key}}">
  <label>{{localize label}}</label>
  <input type="number" name="system.abilities.{{@key}}.value" 
         value="{{ability.value}}" {{#unless @root.isEditable}}disabled{{/unless}}>
  <span class="modifier">{{numberFormat ability.mod sign=true}}</span>
</div>
```

## Helpers Personnalisés

### Créer un helper simple

```javascript
// Dans le hook init
Hooks.once("init", () => {
  
  // Helper pour afficher un modificateur avec couleur
  Handlebars.registerHelper("colorMod", function(value) {
    const color = value >= 0 ? "green" : "red";
    const sign = value >= 0 ? "+" : "";
    return new Handlebars.SafeString(
      `<span style="color: ${color}">${sign}${value}</span>`
    );
  });
  
  // Helper de comparaison personnalisé
  Handlebars.registerHelper("between", function(value, min, max, options) {
    if (value >= min && value <= max) {
      return options.fn(this);
    }
    return options.inverse(this);
  });
  
});
```

### Utilisation

```handlebars
{{! Helper simple }}
<span>Modificateur: {{colorMod actor.system.abilities.str.mod}}</span>

{{! Helper bloc }}
{{#between hp 1 10}}
  <span class="warning">HP faibles !</span>
{{else}}
  <span>HP OK</span>
{{/between}}
```

## Organisation des Templates

### Structure recommandée

```
templates/
├── actor/                    # Sheets d'acteurs
│   ├── character-sheet.hbs
│   ├── npc-sheet.hbs
│   └── parts/                # Parties de sheet
│       ├── actor-header.hbs
│       ├── actor-tabs.hbs
│       ├── actor-attributes.hbs
│       ├── actor-inventory.hbs
│       └── actor-spells.hbs
├── item/                     # Sheets d'items
│   ├── item-sheet.hbs
│   └── parts/
│       ├── item-header.hbs
│       └── item-description.hbs
├── chat/                     # Messages de chat
│   ├── roll-card.hbs
│   └── item-card.hbs
├── dialog/                   # Dialogues
│   └── roll-dialog.hbs
└── partials/                 # Partials globaux
    ├── ability-score.hbs
    └── resource-bar.hbs
```

### Système PARTS (ApplicationV2)

Avec ApplicationV2, les templates sont organisés en PARTS :

```javascript
static PARTS = {
  header: {
    template: "systems/mon-systeme/templates/actor/parts/actor-header.hbs"
  },
  tabs: {
    template: "systems/mon-systeme/templates/actor/parts/actor-tabs.hbs"
  },
  attributes: {
    template: "systems/mon-systeme/templates/actor/parts/actor-attributes.hbs",
    scrollable: [".attributes-list"]
  },
  inventory: {
    template: "systems/mon-systeme/templates/actor/parts/actor-inventory.hbs",
    scrollable: [".inventory-list"]
  }
};
```

## Bonnes Pratiques

### Performance

```handlebars
{{! ❌ Éviter les calculs complexes dans les templates }}
{{#if (and (gt level 5) (eq class "fighter") (not exhausted))}}

{{! ✅ Pré-calculer dans _prepareContext() }}
{{#if canMultiAttack}}
```

### Sécurité

```handlebars
{{! ❌ Dangereux - injection HTML possible }}
{{{userInput}}}

{{! ✅ Sûr - échappé automatiquement }}
{{userInput}}

{{! ✅ Si HTML nécessaire, sanitizer côté JS }}
{{{sanitizedDescription}}}
```

### Maintenabilité

```handlebars
{{! ❌ Template monolithique de 500 lignes }}
<div class="sheet">
  <!-- tout le contenu ici -->
</div>

{{! ✅ Découpage en partials }}
<div class="sheet">
  {{> "path/to/header.hbs"}}
  {{> "path/to/tabs.hbs"}}
  {{> "path/to/content.hbs"}}
</div>
```

## Débogage

### Inspecter le contexte

```handlebars
{{! Afficher tout le contexte (dev seulement) }}
<pre>{{stringify this}}</pre>

{{! Helper personnalisé pour debug }}
{{debug actor.system}}
```

```javascript
// Helper de debug
Handlebars.registerHelper("debug", function(value) {
  console.log("Handlebars Debug:", value);
  return "";
});

Handlebars.registerHelper("stringify", function(value) {
  return JSON.stringify(value, null, 2);
});
```

## Ressources

- [Documentation Handlebars](https://handlebarsjs.com/guide/)
- [Foundry API - Handlebars](https://foundryvtt.com/api/classes/client.HandlebarsHelpers.html)
- [Foundry Wiki - HTML Templates](https://foundryvtt.wiki/en/development/guides/HTML-templates)
