# Système de Dés dans FoundryVTT v13

> Guide complet pour créer et personnaliser les jets de dés dans un système FoundryVTT.

## Vue d'ensemble

Le système de dés de FoundryVTT est construit autour de la classe `Roll`, qui parse des formules de dés, les évalue et affiche les résultats. Les systèmes peuvent étendre ces classes pour créer des comportements personnalisés (avantage/désavantage, critiques, dégâts, etc.).

```
┌─────────────────────────────────────────────────────────────────────┐
│                    ARCHITECTURE DU SYSTÈME DE DÉS                   │
├─────────────────────────────────────────────────────────────────────┤
│                                                                     │
│    ┌──────────────┐                                                 │
│    │  Roll (Core) │  Classe de base Foundry                         │
│    └──────┬───────┘                                                 │
│           │                                                         │
│           ▼                                                         │
│    ┌──────────────┐     ┌─────────────┐     ┌──────────────┐       │
│    │  BasicRoll   │────▶│  D20Roll    │     │  DamageRoll  │       │
│    │  (Système)   │     │  (Système)  │     │  (Système)   │       │
│    └──────────────┘     └─────────────┘     └──────────────┘       │
│           │                    │                    │               │
│           └────────────────────┴────────────────────┘               │
│                                │                                    │
│                                ▼                                    │
│                       ┌────────────────┐                            │
│                       │  ChatMessage   │                            │
│                       │  (avec rolls)  │                            │
│                       └────────────────┘                            │
│                                                                     │
└─────────────────────────────────────────────────────────────────────┘
```

## La Classe Roll (Foundry Core)

### Création d'un jet simple

```javascript
// Syntaxe de base
const roll = new Roll("1d20 + 5");

// Avec des données dynamiques
const roll = new Roll("1d20 + @abilities.str.mod", {
  abilities: { str: { mod: 3 } }
});

// Avec des options
const roll = new Roll("1d20 + @mod", data, {
  flavor: "Jet d'attaque",
  rollMode: "gmroll"
});
```

### Évaluation d'un jet

```javascript
// Évaluation asynchrone (recommandé)
const roll = new Roll("2d6 + 3");
await roll.evaluate();

console.log(roll.total);    // Résultat total (ex: 11)
console.log(roll.result);   // Chaîne de résultat (ex: "5 + 3 + 3")
console.log(roll.formula);  // Formule originale (ex: "2d6 + 3")
console.log(roll.dice);     // Array des DiceTerm

// Évaluation synchrone (v13+)
const roll = new Roll("1d20 + 5");
roll.evaluateSync();

// Options d'évaluation
await roll.evaluate({
  maximize: true,    // Force le maximum sur tous les dés
  minimize: true,    // Force le minimum sur tous les dés
  allowInteractive: true  // Permet les effets visuels
});
```

### Envoi vers le Chat

```javascript
// Méthode simple
const roll = new Roll("1d20 + 5");
await roll.evaluate();
await roll.toMessage({
  speaker: ChatMessage.getSpeaker({ actor: myActor }),
  flavor: "Jet de Dextérité"
});

// Avec plus d'options
await roll.toMessage({
  speaker: ChatMessage.getSpeaker({ actor: myActor }),
  flavor: "Jet d'attaque avec épée longue",
  rollMode: game.settings.get("core", "rollMode"),
  flags: {
    "mon-systeme": {
      type: "attack",
      weaponId: weapon.id
    }
  }
});

// Créer le message sans l'envoyer
const messageData = await roll.toMessage({}, { create: false });
// Modifier les données puis créer manuellement
messageData.content = "Texte personnalisé";
await ChatMessage.create(messageData);
```

## Formules de Dés

### Syntaxe Standard

| Formule | Description |
|---------|-------------|
| `1d20` | Lance 1 dé à 20 faces |
| `2d6` | Lance 2 dés à 6 faces |
| `d20` | Raccourci pour 1d20 |
| `4d6 + 2` | 4d6 plus un modificateur fixe |
| `1d20 + 1d4` | Combinaison de dés |
| `(2d6)kh1` | Garde le plus haut des 2d6 |

### Variables et Références

```javascript
// Référence aux données de l'acteur
const roll = new Roll("1d20 + @abilities.str.mod + @prof", {
  abilities: { str: { mod: 4 } },
  prof: 3
});

// Accès imbriqué
const roll = new Roll("@item.damage.parts[0]", {
  item: {
    damage: {
      parts: ["2d6", "1d8"]
    }
  }
});

// Valeurs par défaut (si la variable n'existe pas)
const roll = new Roll("1d20 + (@bonus || 0)", { bonus: undefined });
```

### Modificateurs de Dés

| Modificateur | Description | Exemple |
|--------------|-------------|---------|
| `kh` / `kh1` | Keep Highest (garder le plus haut) | `2d20kh1` |
| `kl` / `kl1` | Keep Lowest (garder le plus bas) | `2d20kl1` |
| `dh` / `dh1` | Drop Highest (retirer le plus haut) | `4d6dh1` |
| `dl` / `dl1` | Drop Lowest (retirer le plus bas) | `4d6dl1` |
| `r` / `r<2` | Reroll (relancer si condition) | `1d20r<2` |
| `ro` / `ro<2` | Reroll Once (relancer une fois) | `1d20ro<2` |
| `x` / `x>=10` | Exploding (exploser si condition) | `1d6x>=6` |
| `min3` | Minimum par dé | `1d20min10` |
| `max15` | Maximum par dé | `1d20max15` |
| `cs>=18` | Count Successes | `10d10cs>=8` |
| `cf<=2` | Count Failures | `10d10cf<=2` |

```javascript
// Exemples pratiques
const avantage = new Roll("2d20kh1 + @mod", { mod: 5 });  // Avantage
const desavantage = new Roll("2d20kl1 + @mod", { mod: 5 }); // Désavantage
const stats = new Roll("4d6dl1");  // Génération de caractéristique
const halflingLucky = new Roll("1d20ro=1 + @mod", { mod: 3 }); // Halfelin Chanceux
```

## Classes de Termes (Dice Terms)

### Hiérarchie des termes

```
RollTerm (base)
├── DiceTerm
│   ├── Die (dés standard)
│   ├── Coin (pièce)
│   └── FateDie (dés Fate/Fudge)
├── NumericTerm (nombres)
├── OperatorTerm (+, -, *, /)
├── ParentheticalTerm (parenthèses)
├── FunctionTerm (fonctions: floor, ceil, etc.)
├── PoolTerm (pools de dés)
└── StringTerm (chaînes non résolues)
```

### Accéder aux termes d'un Roll

```javascript
const roll = new Roll("2d6 + 1d8 + 5");
await roll.evaluate();

// Tous les termes
console.log(roll.terms);
// [Die(2d6), OperatorTerm(+), Die(1d8), OperatorTerm(+), NumericTerm(5)]

// Uniquement les dés
console.log(roll.dice);
// [Die(2d6), Die(1d8)]

// Résultats individuels d'un dé
const d6 = roll.dice[0];
console.log(d6.results);
// [{result: 4, active: true}, {result: 2, active: true}]
console.log(d6.total);  // 6
```

## Classes de Rolls Personnalisées (Exemple dnd5e)

### BasicRoll - Classe de base système

Le système dnd5e étend `Roll` avec `BasicRoll` pour ajouter un workflow standardisé :

```javascript
// Structure de BasicRoll (dnd5e/module/dice/basic-roll.mjs)
export default class BasicRoll extends Roll {
  
  // Application de configuration par défaut
  static DefaultConfigurationDialog = RollConfigurationDialog;

  /**
   * Workflow complet : configuration → évaluation → message
   */
  static async build(config={}, dialog={}, message={}) {
    const rolls = await this.buildConfigure(config, dialog, message);
    await this.buildEvaluate(rolls, config, message);
    await this.buildPost(rolls, config, message);
    return rolls;
  }

  /**
   * Créer un roll depuis une configuration
   */
  static fromConfig(config, process) {
    const formula = (config.parts ?? []).join(" + ");
    config.options ??= {};
    config.options.target ??= process.target;
    return new this(formula, config.data, config.options);
  }

  /**
   * Propriétés de succès/échec basées sur une cible
   */
  get isFailure() {
    if ( !this._evaluated ) return;
    if ( !Number.isNumeric(this.options.target) ) return false;
    return this.total < this.options.target;
  }

  get isSuccess() {
    if ( !this._evaluated ) return;
    if ( !Number.isNumeric(this.options.target) ) return false;
    return this.total >= this.options.target;
  }

  /**
   * Créer un message de chat avec plusieurs rolls
   */
  static async toMessage(rolls, messageData={}, { rollMode, create=true }={}) {
    for ( const roll of rolls ) {
      if ( !roll._evaluated ) await roll.evaluate();
    }
    messageData.rolls = rolls;
    // ... création du message
  }
}
```

### D20Roll - Jets de d20 avec avantage/désavantage

```javascript
// Structure de D20Roll (dnd5e/module/dice/d20-roll.mjs)
export default class D20Roll extends BasicRoll {
  
  // Modes d'avantage
  static ADV_MODE = {
    NORMAL: 0,
    ADVANTAGE: 1,
    DISADVANTAGE: -1
  };

  constructor(formula, data, options) {
    super(formula, data, options);
    this.#createD20Die();
    if ( !this.options.configured ) this.configureModifiers();
  }

  /**
   * Référence au dé d20 principal
   */
  get d20() {
    return this.terms[0];
  }

  /**
   * Propriétés de critique
   */
  get isCritical() {
    return this.d20.isCriticalSuccess;
  }

  get isFumble() {
    return this.d20.isCriticalFailure;
  }

  get hasAdvantage() {
    return this.options.advantageMode === D20Roll.ADV_MODE.ADVANTAGE;
  }

  get hasDisadvantage() {
    return this.options.advantageMode === D20Roll.ADV_MODE.DISADVANTAGE;
  }

  /**
   * Configuration automatique des modificateurs
   */
  configureModifiers() {
    if ( !this.validD20Roll ) return;

    // Déterminer le mode d'avantage
    const { advantage, disadvantage } = this.options;
    if ( advantage && !disadvantage ) {
      this.options.advantageMode = D20Roll.ADV_MODE.ADVANTAGE;
    }
    else if ( !advantage && disadvantage ) {
      this.options.advantageMode = D20Roll.ADV_MODE.DISADVANTAGE;
    }

    // Appliquer les flags au d20
    this.d20.applyAdvantage(this.options.advantageMode);
    this.d20.applyFlag("halflingLucky", this.options.halflingLucky);
    
    // Reliable Talent : minimum 10 sur le d20
    if ( this.options.reliableTalent ) {
      this.d20.applyRange({ minimum: 10 });
    }

    this.resetFormula();
    this.options.configured = true;
  }
}
```

### Utilisation de D20Roll

```javascript
// Jet d'attaque avec avantage
const attackRoll = await D20Roll.build({
  rolls: [{
    parts: ["@mod", "@prof"],
    data: { mod: 5, prof: 3 },
    options: { advantage: true }
  }],
  hookNames: ["Attack"]
}, {
  configure: true  // Afficher la boîte de dialogue
}, {
  create: true,
  data: {
    speaker: ChatMessage.getSpeaker({ actor }),
    flavor: "Jet d'attaque - Épée longue",
    "flags.dnd5e.roll.type": "attack"
  }
});

// Jet de sauvegarde
const saveRoll = await D20Roll.build({
  rolls: [{
    parts: ["@abilities.con.save"],
    data: actor.getRollData(),
    options: { target: 15 }  // DC de la sauvegarde
  }]
}, { configure: true }, {
  create: true,
  data: {
    speaker: ChatMessage.getSpeaker({ actor }),
    flavor: "Jet de sauvegarde de Constitution (DC 15)"
  }
});
```

### D20Die - Dé d20 personnalisé

```javascript
// Structure de D20Die (dnd5e/module/dice/d20-die.mjs)
export default class D20Die extends Die {
  
  static CRITICAL_SUCCESS_TOTAL = 20;
  static CRITICAL_FAILURE_TOTAL = 1;

  constructor({ number=1, faces=20, ...args }={}) {
    super({ number, faces, ...args });
  }

  get isCriticalSuccess() {
    if ( !this.isValid || !this._evaluated ) return;
    return this.total >= this.options.criticalSuccess;
  }

  get isCriticalFailure() {
    if ( !this.isValid || !this._evaluated ) return;
    return this.total <= this.options.criticalFailure;
  }

  /**
   * Appliquer le mode d'avantage
   */
  applyAdvantage(advantageMode) {
    this.modifiers.findSplice(m => ["kh", "kl"].includes(m));
    
    if ( advantageMode === CONFIG.Dice.D20Roll.ADV_MODE.NORMAL ) {
      this.number = 1;
    } else {
      const isAdvantage = advantageMode === CONFIG.Dice.D20Roll.ADV_MODE.ADVANTAGE;
      this.number = (isAdvantage && this.options.elvenAccuracy) ? 3 : 2;
      this.modifiers.push(isAdvantage ? "kh" : "kl");
    }
  }

  /**
   * Appliquer des flags spéciaux
   */
  applyFlag(flag, enabled) {
    this.options[flag] = enabled;
    
    // Halfling Lucky : relancer un 1 une fois
    if ( flag === "halflingLucky" ) {
      const index = this.modifiers.findIndex(m => m === "r1=1");
      if ( enabled && (index === -1) ) this.modifiers.push("r1=1");
      else if ( !enabled && (index !== -1) ) this.modifiers.splice(index, 1);
    }
  }

  /**
   * Appliquer min/max
   */
  applyRange({ minimum, maximum }) {
    if ( minimum ) this.modifiers.push(`min${minimum}`);
    if ( maximum ) this.modifiers.push(`max${maximum}`);
  }
}
```

### DamageRoll - Jets de dégâts avec critiques

```javascript
// Structure de DamageRoll (dnd5e/module/dice/damage-roll.mjs)
export default class DamageRoll extends BasicRoll {

  constructor(formula, data, options) {
    super(formula, data, options);
    if ( !this.options.preprocessed ) this.preprocessFormula();
    if ( !this.options.configured ) this.configureDamage();
  }

  get isCritical() {
    return this.options.isCritical === true;
  }

  /**
   * Configuration des dégâts critiques
   */
  configureDamage({ critical={} }={}) {
    critical = foundry.utils.mergeObject(critical, this.options.critical ?? {});

    for ( let term of this.terms ) {
      if ( term instanceof DiceTerm ) {
        term.options.baseNumber = term.options.baseNumber ?? term.number;
        term.number = term.options.baseNumber;
        
        if ( this.isCritical ) {
          const multiplier = critical.multiplier ?? 2;
          const bonusDice = critical.bonusDice ?? 0;
          term.alter(multiplier, bonusDice);
          term.options.critical = true;
        }
      }
    }

    // Ajouter les dégâts bonus critiques
    if ( this.isCritical && critical.bonusDamage ) {
      const extraTerms = new Roll(critical.bonusDamage, this.data).terms;
      this.terms.push(...extraTerms);
    }

    this.resetFormula();
    this.options.configured = true;
  }
}
```

### Utilisation de DamageRoll

```javascript
// Jet de dégâts simple
const damageRoll = await DamageRoll.build({
  rolls: [{
    parts: ["2d6", "@mod"],
    data: { mod: 4 },
    options: { type: "slashing" }
  }]
}, { configure: true }, {
  create: true,
  data: {
    speaker: ChatMessage.getSpeaker({ actor }),
    flavor: "Dégâts - Épée longue",
    "flags.dnd5e.roll.type": "damage"
  }
});

// Jet de dégâts critique
const criticalDamage = await DamageRoll.build({
  rolls: [{
    parts: ["2d6", "@mod"],
    data: { mod: 4 },
    options: {
      type: "slashing",
      isCritical: true,
      critical: {
        multiplier: 2,
        bonusDice: 1,  // Dé bonus sur les critiques brutaux
        bonusDamage: "1d8"  // Dégâts bonus supplémentaires
      }
    }
  }]
}, { configure: false }, { create: true });
```

## Enregistrement des Classes Personnalisées

### Dans le hook `init`

```javascript
Hooks.once("init", () => {
  // Enregistrer les classes de rolls
  CONFIG.Dice.BasicRoll = BasicRoll;
  CONFIG.Dice.D20Roll = D20Roll;
  CONFIG.Dice.DamageRoll = DamageRoll;
  
  // Enregistrer le dé d20 personnalisé
  CONFIG.Dice.D20Die = D20Die;
  
  // Enregistrer un terme de dé personnalisé
  CONFIG.Dice.terms.d = D20Die;  // Remplace le 'd' standard
  
  // Ou ajouter un nouveau type de dé
  CONFIG.Dice.terms.f = FateDie;  // Pour les dés Fate (df)
});
```

## Utilitaires de Formules

### simplifyRollFormula

Simplifie une formule de dés en combinant les termes compatibles :

```javascript
// dnd5e/module/dice/simplify-roll-formula.mjs
import simplifyRollFormula from "./simplify-roll-formula.mjs";

// Simplification basique
simplifyRollFormula("1d6 + 2 + 3");  // "1d6 + 5"

// Combinaison de dés identiques
simplifyRollFormula("1d6 + 1d6 + 2");  // "2d6 + 2"

// Options
simplifyRollFormula("1d6[fire] + 2", { preserveFlavor: true });  // Garde les annotations

// Extraire uniquement les termes déterministes
simplifyRollFormula("1d20 + 5 + @mod", { deterministic: true });  // "5"
```

### aggregateDamageRolls

Combine plusieurs jets de dégâts par type :

```javascript
// dnd5e/module/dice/aggregate-damage-rolls.mjs
import aggregateDamageRolls from "./aggregate-damage-rolls.mjs";

const rolls = [
  new DamageRoll("2d6", {}, { type: "slashing" }),
  new DamageRoll("1d6", {}, { type: "fire" }),
  new DamageRoll("1d6", {}, { type: "slashing" })
];

// Agrège par type de dégâts
const aggregated = aggregateDamageRolls(rolls);
// Résultat : [DamageRoll "3d6" (slashing), DamageRoll "1d6" (fire)]

// Respecter aussi les propriétés (magique, argenté, etc.)
const aggregatedWithProps = aggregateDamageRolls(rolls, { respectProperties: true });
```

## Boîtes de Dialogue de Configuration

### RollConfigurationDialog

Le système dnd5e fournit une boîte de dialogue configurable pour les jets :

```javascript
// Afficher une boîte de dialogue de configuration
const rolls = await RollConfigurationDialog.configure(
  // Configuration du processus
  {
    rolls: [{
      parts: ["1d20", "@mod"],
      data: { mod: 5 },
      options: {}
    }],
    target: 15  // DC optionnel
  },
  // Configuration de la boîte de dialogue
  {
    configure: true,
    applicationClass: CustomConfigDialog,  // Classe personnalisée optionnelle
    options: {
      rollType: D20Roll,
      default: { rollMode: "gmroll" }
    }
  },
  // Configuration du message
  {
    create: true,
    rollMode: "publicroll",
    data: { flavor: "Mon jet personnalisé" }
  }
);
```

### Structure de la boîte de dialogue

```javascript
// Personnaliser la boîte de dialogue
class CustomRollDialog extends RollConfigurationDialog {
  
  static DEFAULT_OPTIONS = {
    ...super.DEFAULT_OPTIONS,
    classes: ["roll-configuration", "custom-dialog"],
    window: {
      title: "Configuration du jet",
      icon: "fa-solid fa-dice-d20"
    }
  };

  static PARTS = {
    formulas: { template: "systems/mon-systeme/templates/dice/formulas.hbs" },
    configuration: { template: "systems/mon-systeme/templates/dice/config.hbs" },
    buttons: { template: "systems/mon-systeme/templates/dice/buttons.hbs" }
  };

  async _prepareConfigurationContext(context, options) {
    context = await super._prepareConfigurationContext(context, options);
    // Ajouter des champs personnalisés
    context.fields.push({
      field: new foundry.data.fields.BooleanField({ label: "Bonus situationnel" }),
      name: "situational",
      value: false
    });
    return context;
  }
}
```

## Hooks du Système de Dés

### Hooks de pré-jet

```javascript
// Avant configuration du jet
Hooks.on("dnd5e.preRoll", (config, dialog, message) => {
  // Modifier la configuration avant le jet
  config.rolls[0].parts.push("@bonus");
  config.rolls[0].data.bonus = 2;
  
  // Retourner false pour annuler le jet
  return true;
});

// Hook spécifique par type
Hooks.on("dnd5e.preRollAttack", (config, dialog, message) => {
  console.log("Jet d'attaque en préparation");
});

Hooks.on("dnd5e.preRollSkill", (config, dialog, message) => {
  console.log("Jet de compétence en préparation");
});
```

### Hooks de post-configuration

```javascript
// Après configuration mais avant évaluation
Hooks.on("dnd5e.postRollConfiguration", (rolls, config, dialog, message) => {
  for ( const roll of rolls ) {
    console.log("Formule configurée:", roll.formula);
  }
  // Retourner false pour annuler
  return true;
});
```

### Hooks de construction de configuration

```javascript
// Lors de la construction de chaque roll
Hooks.on("dnd5e.buildRollConfig", (app, config, formData, index) => {
  // Ajouter des parties basées sur les données du formulaire
  if ( formData?.get("situational") ) {
    config.parts.push("@situational");
    config.data.situational = formData.get("situationalValue");
  }
});
```

## Intégration avec les Messages de Chat

### Affichage des résultats

```javascript
// Dans ChatMessage5e (dnd5e/module/documents/chat-message.mjs)

// Mise en évidence des critiques
_highlightCriticalSuccessFailure(html) {
  for ( let [index, d20Roll] of this.rolls.entries() ) {
    const total = html.querySelectorAll(".dice-total")[index];
    
    if ( d20Roll.isCritical ) total.classList.add("critical");
    if ( d20Roll.isFumble ) total.classList.add("fumble");
    if ( d20Roll.isSuccess ) total.classList.add("success");
    if ( d20Roll.isFailure ) total.classList.add("failure");
  }
}

// Tooltip de dégâts enrichi
_enrichDamageTooltip(rolls, html) {
  const aggregated = aggregateDamageRolls(rolls);
  // Afficher les dégâts groupés par type
}
```

### Application des dégâts

```javascript
// Appliquer les dégâts aux tokens sélectionnés
applyChatCardDamage(li, multiplier) {
  const damages = aggregateDamageRolls(this.rolls, { respectProperties: true })
    .map(roll => ({
      value: roll.total * multiplier,
      type: roll.options.type,
      properties: new Set(roll.options.properties ?? [])
    }));
  
  return Promise.all(canvas.tokens.controlled.map(t => {
    return t.actor?.applyDamage(damages, { multiplier, isDelta: true });
  }));
}
```

## Exemples Complets

### Jet de compétence avec toutes les options

```javascript
async function rollSkillCheck(actor, skillId, options = {}) {
  const skill = actor.system.skills[skillId];
  const ability = actor.system.abilities[skill.ability];
  
  const config = {
    rolls: [{
      parts: ["@mod"],
      data: {
        mod: skill.total
      },
      options: {
        advantage: options.advantage,
        disadvantage: options.disadvantage,
        reliableTalent: skill.value >= 2 && actor.flags.dnd5e?.reliableTalent,
        halflingLucky: actor.flags.dnd5e?.halflingLucky,
        target: options.dc
      }
    }],
    hookNames: ["Skill", "AbilityCheck"]
  };

  const dialog = {
    configure: !options.fastForward,
    options: {
      rollType: CONFIG.Dice.D20Roll
    }
  };

  const message = {
    create: true,
    data: {
      speaker: ChatMessage.getSpeaker({ actor }),
      flavor: `${CONFIG.DND5E.skills[skillId].label} (${CONFIG.DND5E.abilities[skill.ability].label})`,
      "flags.dnd5e.roll.type": "skill",
      "flags.dnd5e.roll.skillId": skillId
    }
  };

  return D20Roll.build(config, dialog, message);
}
```

### Jet d'attaque complet

```javascript
async function rollAttack(actor, weapon, options = {}) {
  const rollData = actor.getRollData();
  rollData.item = weapon.getRollData();
  
  const parts = [];
  
  // Modificateur d'attaque
  parts.push("@mod");
  rollData.mod = weapon.system.attackBonus + 
                 actor.system.abilities[weapon.system.ability].mod;
  
  // Bonus de maîtrise
  if ( weapon.system.proficient ) {
    parts.push("@prof");
    rollData.prof = actor.system.attributes.prof;
  }

  const config = {
    rolls: [{
      parts,
      data: rollData,
      options: {
        advantage: options.advantage,
        disadvantage: options.disadvantage,
        criticalSuccess: 20,
        criticalFailure: 1
      }
    }],
    hookNames: ["Attack"]
  };

  const dialog = {
    configure: !options.fastForward
  };

  const message = {
    create: true,
    data: {
      speaker: ChatMessage.getSpeaker({ actor }),
      flavor: `${weapon.name} - Jet d'attaque`,
      "flags.dnd5e.roll.type": "attack",
      "flags.dnd5e.item.uuid": weapon.uuid
    }
  };

  const rolls = await D20Roll.build(config, dialog, message);
  
  // Si critique, proposer les dégâts critiques
  if ( rolls[0]?.isCritical && options.autoDamage ) {
    await rollDamage(actor, weapon, { isCritical: true });
  }
  
  return rolls;
}
```

### Jet de dégâts avec types multiples

```javascript
async function rollDamage(actor, weapon, options = {}) {
  const rollData = actor.getRollData();
  const rolls = [];
  
  // Parcourir les parties de dégâts
  for ( const [formula, type] of weapon.system.damage.parts ) {
    rolls.push({
      parts: [formula, "@mod"],
      data: {
        ...rollData,
        mod: actor.system.abilities[weapon.system.ability].mod
      },
      options: {
        type: type,
        isCritical: options.isCritical,
        critical: {
          multiplier: 2,
          bonusDice: actor.flags.dnd5e?.brutalCritical ? 1 : 0
        },
        properties: Array.from(weapon.system.properties)
      }
    });
  }

  const config = {
    rolls,
    hookNames: ["Damage"],
    isCritical: options.isCritical
  };

  return DamageRoll.build(config, { configure: !options.fastForward }, {
    create: true,
    data: {
      speaker: ChatMessage.getSpeaker({ actor }),
      flavor: `${weapon.name} - ${options.isCritical ? "Dégâts critiques" : "Dégâts"}`,
      "flags.dnd5e.roll.type": "damage"
    }
  });
}
```

## Types de Configuration (TypeScript/JSDoc)

Les types principaux utilisés pour la configuration des rolls :

```javascript
/**
 * @typedef BasicRollProcessConfiguration
 * @property {BasicRollConfiguration[]} rolls  Configuration des rolls individuels
 * @property {boolean} [evaluate=true]         Évaluer les rolls ?
 * @property {Event} [event]                   Événement déclencheur
 * @property {string[]} [hookNames]            Suffixes pour les hooks
 * @property {Document} [subject]              Document source
 * @property {number} [target]                 Valeur cible par défaut
 */

/**
 * @typedef BasicRollConfiguration
 * @property {string[]} [parts=[]]         Parties de la formule
 * @property {object} [data={}]            Données pour les variables
 * @property {boolean} [situational=true]  Autoriser bonus situationnel ?
 * @property {BasicRollOptions} [options]  Options du roll
 */

/**
 * @typedef D20RollOptions
 * @property {boolean} [advantage]         Avantage ?
 * @property {boolean} [disadvantage]      Désavantage ?
 * @property {number} [criticalSuccess]    Seuil de critique (défaut: 20)
 * @property {number} [criticalFailure]    Seuil d'échec critique (défaut: 1)
 * @property {boolean} [elvenAccuracy]     Précision elfique (3 dés)
 * @property {boolean} [halflingLucky]     Chance halfeline (relance 1)
 * @property {number} [minimum]            Minimum sur le d20
 * @property {number} [maximum]            Maximum sur le d20
 */

/**
 * @typedef CriticalDamageConfiguration
 * @property {boolean} [allow=true]        Autoriser les critiques ?
 * @property {number} [multiplier=2]       Multiplicateur de dégâts
 * @property {number} [bonusDice=0]        Dés bonus sur critique
 * @property {string} [bonusDamage]        Formule de dégâts bonus
 * @property {boolean} [multiplyDice]      Multiplier les résultats ?
 * @property {boolean} [multiplyNumeric]   Multiplier les numériques ?
 */
```

## Bonnes Pratiques

1. **Toujours évaluer de manière asynchrone** sauf si absolument nécessaire
2. **Utiliser `build()` pour le workflow complet** - gère configuration, évaluation et message
3. **Stocker les métadonnées dans les flags** du message pour référence ultérieure
4. **Tester les formules** avec `Roll.validate(formula)` avant évaluation
5. **Prévoir les hooks** pour permettre aux modules de modifier les jets
6. **Gérer les cas d'annulation** (dialog fermé, hook retourne false)

---

*Documentation suivante : [07-applications.md](./07-applications.md) - Interfaces utilisateur avec Application V2*
