# Système de Hooks

> Documentation technique pour FoundryVTT v13

## Vue d'ensemble

Les **Hooks** sont le système d'événements de FoundryVTT. Ils permettent aux systèmes et modules d'intercepter et de réagir aux événements du cycle de vie de l'application. C'est le mécanisme principal d'extension et de personnalisation.

```
┌─────────────────────────────────────────────────────────────────┐
│                    ARCHITECTURE DES HOOKS                       │
├─────────────────────────────────────────────────────────────────┤
│                                                                 │
│   Foundry Core    ──────▶  Hooks.callAll("eventName")           │
│                                    │                            │
│                                    ▼                            │
│                    ┌───────────────────────────────┐            │
│                    │   File d'attente des listeners│            │
│                    ├───────────────────────────────┤            │
│                    │  listener1() ──▶ exécuté      │            │
│                    │  listener2() ──▶ exécuté      │            │
│                    │  listener3() ──▶ exécuté      │            │
│                    └───────────────────────────────┘            │
│                                                                 │
│   Système/Module  ──────▶  Hooks.on("eventName", callback)      │
│                                                                 │
└─────────────────────────────────────────────────────────────────┘
```

## API des Hooks

### Hooks.on(name, callback)

Enregistre un listener permanent qui sera appelé à chaque fois que le hook est déclenché.

```javascript
// Écouter un hook de façon permanente
Hooks.on("updateActor", (actor, changes, options, userId) => {
  console.log(`${actor.name} a été modifié`);
});

// Retourne un ID pour pouvoir désinscrire le listener
const hookId = Hooks.on("renderActorSheet", (sheet, html, data) => {
  // Modifier le DOM de la feuille
});

// Désinscrire le listener
Hooks.off("renderActorSheet", hookId);
```

### Hooks.once(name, callback)

Enregistre un listener qui ne sera exécuté qu'**une seule fois**, puis automatiquement désactivé.

```javascript
// Exécuté une seule fois au démarrage
Hooks.once("ready", () => {
  console.log("Le monde est chargé !");
  // Initialisation unique
});

// Utile pour attendre un événement spécifique
Hooks.once("renderChatMessage", (message) => {
  if ( message.id === targetMessageId ) {
    // Réagir au rendu de CE message spécifique
  }
});
```

### Hooks.call(name, ...args)

Déclenche un hook et permet aux listeners de **bloquer l'exécution** en retournant `false`.

```javascript
// Déclencher un hook qui peut être annulé
const allowed = Hooks.call("dnd5e.preShortRest", actor, config);
if ( allowed === false ) {
  // Un listener a bloqué le repos court
  return;
}

// Le listener peut bloquer
Hooks.on("dnd5e.preShortRest", (actor, config) => {
  if ( actor.system.attributes.hp.value === 0 ) {
    ui.notifications.warn("Impossible de se reposer étant inconscient !");
    return false; // Bloque le repos
  }
});
```

### Hooks.callAll(name, ...args)

Déclenche un hook et exécute **tous les listeners** sans possibilité de blocage.

```javascript
// Déclencher un hook informatif (ne peut pas être bloqué)
Hooks.callAll("dnd5e.shortRest", actor, result);

// Tous les listeners sont exécutés, même si l'un retourne false
Hooks.on("dnd5e.shortRest", (actor, result) => {
  console.log("Repos terminé, notifications...");
  return false; // N'a aucun effet de blocage
});
```

## Comparaison call vs callAll

| Aspect | `Hooks.call()` | `Hooks.callAll()` |
|--------|----------------|-------------------|
| Blocage | Peut être bloqué par `return false` | Ne peut pas être bloqué |
| Usage | Événements `pre-` (avant action) | Événements `post-` (après action) |
| Retour | `false` si bloqué, sinon `true` | Toujours `true` |
| Exemple | `preCreateActor`, `preUpdate*` | `createActor`, `update*`, `render*` |

## Hooks Synchrones vs Asynchrones

### Hooks Synchrones (par défaut)

Les callbacks sont exécutés **immédiatement** et dans l'ordre d'enregistrement.

```javascript
Hooks.on("preCreateActor", (document, data, options, userId) => {
  // Exécution synchrone
  document.updateSource({ name: data.name.toUpperCase() });
  // Le document est modifié AVANT la création
});
```

### Callbacks Asynchrones

Les callbacks peuvent être `async`, mais Foundry **n'attend pas** leur résolution.

```javascript
// ⚠️ ATTENTION : Foundry n'attend pas la Promise
Hooks.on("createActor", async (actor, options, userId) => {
  // Cette opération asynchrone peut ne pas être terminée
  // avant que d'autres hooks soient appelés
  await someAsyncOperation();
});

// ✅ SOLUTION : Utiliser les méthodes _preCreate/_onCreate du Document
class Actor5e extends Actor {
  async _preCreate(data, options, user) {
    // Foundry attend cette Promise
    await super._preCreate(data, options, user);
    await this.doSomethingAsync();
  }
}
```

## Ordre d'Exécution

Les listeners sont exécutés dans l'ordre d'enregistrement :

```
┌─────────────────────────────────────────────────────────────────┐
│                    ORDRE D'EXÉCUTION                            │
├─────────────────────────────────────────────────────────────────┤
│                                                                 │
│  1. Hooks.on() du Core Foundry (en premier)                     │
│  2. Hooks.on() des systèmes (ordre de chargement)               │
│  3. Hooks.on() des modules (ordre alphabétique du manifest)     │
│  4. Hooks.once() dans le même ordre                             │
│                                                                 │
│  ⚠️ Les Hooks.once() sont retirés après exécution               │
│                                                                 │
└─────────────────────────────────────────────────────────────────┘
```

## Hooks du Cycle de Vie Système

Ces hooks sont exécutés une seule fois au démarrage, dans cet ordre précis :

### 1. Hook `init`

Premier hook, déclenché **avant** le chargement des données du monde.

```javascript
Hooks.once("init", function() {
  // ✅ Ce qui doit être fait ici :
  
  // 1. Exposer le système globalement
  globalThis.dnd5e = game.dnd5e = Object.assign(game.system, globalThis.dnd5e);
  
  // 2. Enregistrer la configuration
  CONFIG.DND5E = DND5E;
  
  // 3. Enregistrer les classes de documents
  CONFIG.Actor.documentClass = documents.Actor5e;
  CONFIG.Item.documentClass = documents.Item5e;
  CONFIG.ActiveEffect.documentClass = documents.ActiveEffect5e;
  CONFIG.ChatMessage.documentClass = documents.ChatMessage5e;
  CONFIG.Combat.documentClass = documents.Combat5e;
  CONFIG.Combatant.documentClass = documents.Combatant5e;
  CONFIG.Token.documentClass = documents.TokenDocument5e;
  CONFIG.User.documentClass = documents.User5e;
  
  // 4. Enregistrer les DataModels
  CONFIG.Actor.dataModels = dataModels.actor.config;
  CONFIG.Item.dataModels = dataModels.item.config;
  CONFIG.ActiveEffect.dataModels = dataModels.activeEffect.config;
  
  // 5. Enregistrer les classes de dés
  CONFIG.Dice.BasicRoll = dice.BasicRoll;
  CONFIG.Dice.D20Roll = dice.D20Roll;
  CONFIG.Dice.DamageRoll = dice.DamageRoll;
  
  // 6. Enregistrer les feuilles de personnage
  const DocumentSheetConfig = foundry.applications.apps.DocumentSheetConfig;
  DocumentSheetConfig.registerSheet(Actor, "dnd5e", CharacterActorSheet, {
    types: ["character"],
    makeDefault: true,
    label: "DND5E.SheetClass.Character"
  });
  
  // 7. Enregistrer les paramètres système
  registerSystemSettings();
  registerSystemKeybindings();
  
  // 8. Précharger les templates Handlebars
  utils.preloadHandlebarsTemplates();
  
  // 9. Enregistrer les enrichers personnalisés
  enrichers.registerCustomEnrichers();
  
  // ❌ À NE PAS FAIRE ici :
  // - Accéder à game.actors, game.items (non disponibles)
  // - Manipuler des données du monde
  // - Effectuer des migrations
});
```

**Ce qui est disponible dans `init` :**
- `game.system` - Informations du système
- `game.settings` - API des paramètres (mais pas les valeurs du monde)
- `CONFIG` - Objet de configuration global
- `Hooks` - Système de hooks

**Ce qui n'est PAS disponible :**
- `game.actors`, `game.items`, `game.scenes`
- `game.users` (liste complète)
- `game.world`

### 2. Hook `i18nInit`

Déclenché après le chargement du système d'internationalisation.

```javascript
Hooks.once("i18nInit", () => {
  // ✅ Ce qui doit être fait ici :
  
  // 1. Configurer les effets de statut (après init, avant prélocalisation)
  _configureStatusEffects();
  
  // 2. Appliquer les règles legacy si nécessaire
  if ( game.settings.get("dnd5e", "rulesVersion") === "legacy" ) {
    const { translations } = game.i18n;
    foundry.utils.mergeObject(translations, {
      "TYPES.Item": {
        race: game.i18n.localize("TYPES.Item.raceLegacy")
      }
    });
  }
  
  // 3. Pré-localiser la configuration
  utils.performPreLocalization(CONFIG.DND5E);
  
  // 4. Localiser les types de données
  Object.values(CONFIG.DND5E.activityTypes).forEach(c => c.documentClass.localize());
  Object.values(CONFIG.DND5E.advancementTypes).forEach(c => c.documentClass.localize());
});
```

**Ce qui est disponible :**
- `game.i18n` - Système de traduction complet
- `game.i18n.localize()`, `game.i18n.format()`
- Toutes les traductions chargées

### 3. Hook `setup`

Déclenché après que tous les systèmes et modules ont terminé leur `init`.

```javascript
Hooks.once("setup", function() {
  // ✅ Ce qui doit être fait ici :
  
  // 1. Configurer les attributs trackables
  _configureTrackableAttributes();
  _configureConsumableAttributes();
  
  // 2. Enregistrer l'art des modules (module art)
  game.dnd5e.moduleArt.registerModuleArt();
  
  // 3. Activer les tooltips
  Tooltips5e.activateListeners();
  game.dnd5e.tooltips.observe();
  
  // 4. Enregistrer les paramètres différés (après que les modules ont pu ajouter les leurs)
  registerDeferredSettings();
  
  // 5. Configurer les packs de compendiums
  setupModulePacks();
  
  // 6. Créer du CSS dynamique
  const style = document.createElement("style");
  style.innerHTML = `/* CSS généré dynamiquement */`;
  document.head.append(style);
});
```

**Ce qui est disponible :**
- Configurations de tous les modules
- Paramètres des modules
- Index des compendiums

**Ce qui n'est PAS encore disponible :**
- `game.actors`, `game.items` (données du monde)

### 4. Hook `ready`

Déclenché quand le monde est **entièrement chargé** et prêt à être utilisé.

```javascript
Hooks.once("ready", function() {
  // ✅ Ce qui doit être fait ici :
  
  // 1. Enregistrer les hooks d'interaction utilisateur
  Hooks.on("hotbarDrop", (bar, data, slot) => {
    if ( ["ActiveEffect", "Activity", "Item"].includes(data.type) ) {
      documents.macro.create5eMacro(data, slot);
      return false;
    }
  });
  
  // 2. Initialiser les registres d'items
  dnd5e.registry.classes.initialize();
  dnd5e.registry.subclasses.initialize();
  
  // 3. Activer les listeners de messages de chat
  documents.ChatMessage5e.activateListeners();
  
  // 4. Initialiser les UI persistantes
  game.dnd5e.bastion.initializeUI();
  
  // 5. Afficher les HUD personnalisés
  if ( CONFIG.DND5E.calendar.application ) {
    dnd5e.ui.calendar = new CONFIG.DND5E.calendar.application();
    dnd5e.ui.calendar.render({ force: true });
  }
  
  // 6. Effectuer les migrations (GM uniquement)
  if ( !game.user.isGM ) return;
  
  const cv = game.settings.get("dnd5e", "systemMigrationVersion");
  const totalDocuments = game.actors.size + game.scenes.size + game.items.size;
  
  if ( !cv && totalDocuments === 0 ) {
    return game.settings.set("dnd5e", "systemMigrationVersion", game.system.version);
  }
  
  if ( cv && !foundry.utils.isNewerVersion(game.system.flags.needsMigrationVersion, cv) ) {
    return;
  }
  
  migrations.migrateWorld();
});
```

**Ce qui est MAINTENANT disponible :**
- `game.actors` - Collection des acteurs
- `game.items` - Collection des items
- `game.scenes` - Collection des scènes
- `game.users` - Collection des utilisateurs
- `game.world` - Données du monde
- `canvas` - Le canvas de jeu (si une scène est active)

## Hooks de Documents

Ces hooks sont déclenchés lors des opérations CRUD sur les documents.

### Pattern pré/post

```
┌─────────────────────────────────────────────────────────────────┐
│                    CYCLE CRUD D'UN DOCUMENT                     │
├─────────────────────────────────────────────────────────────────┤
│                                                                 │
│  CREATE:                                                        │
│    preCreate[Document] ──▶ Peut bloquer avec return false       │
│           │                                                     │
│           ▼                                                     │
│    create[Document]    ──▶ Informationnel (post-création)       │
│                                                                 │
│  UPDATE:                                                        │
│    preUpdate[Document] ──▶ Peut modifier/bloquer                │
│           │                                                     │
│           ▼                                                     │
│    update[Document]    ──▶ Informationnel (post-modification)   │
│                                                                 │
│  DELETE:                                                        │
│    preDelete[Document] ──▶ Peut bloquer la suppression          │
│           │                                                     │
│           ▼                                                     │
│    delete[Document]    ──▶ Informationnel (post-suppression)    │
│                                                                 │
└─────────────────────────────────────────────────────────────────┘
```

### Hooks de Création

```javascript
// preCreateActor - AVANT création, peut bloquer
Hooks.on("preCreateActor", (document, data, options, userId) => {
  // Modifier les données avant création
  document.updateSource({ 
    "system.attributes.hp.value": 10 
  });
  
  // Bloquer la création si conditions non remplies
  if ( !game.user.isGM && data.type === "npc" ) {
    ui.notifications.warn("Seul le MJ peut créer des PNJ");
    return false;
  }
});

// createActor - APRÈS création
Hooks.on("createActor", (document, options, userId) => {
  // Notification, logs, effets secondaires
  console.log(`Acteur ${document.name} créé par ${game.users.get(userId).name}`);
  
  // Créer des items par défaut
  if ( document.type === "character" && document.isOwner ) {
    // Ajouter l'équipement de départ
  }
});
```

### Hooks de Modification

```javascript
// preUpdateActor - AVANT modification
Hooks.on("preUpdateActor", (document, changes, options, userId) => {
  // Vérifier/modifier les changements
  const newHp = foundry.utils.getProperty(changes, "system.attributes.hp.value");
  const maxHp = document.system.attributes.hp.max;
  
  if ( newHp !== undefined && newHp > maxHp ) {
    // Plafonner les PV au maximum
    foundry.utils.setProperty(changes, "system.attributes.hp.value", maxHp);
  }
});

// updateActor - APRÈS modification
Hooks.on("updateActor", (document, changes, options, userId) => {
  // Réagir aux changements
  if ( foundry.utils.hasProperty(changes, "system.attributes.hp.value") ) {
    const hp = document.system.attributes.hp;
    if ( hp.value === 0 ) {
      ui.notifications.info(`${document.name} est tombé inconscient !`);
    }
  }
});
```

### Hooks de Suppression

```javascript
// preDeleteActor - AVANT suppression
Hooks.on("preDeleteActor", (document, options, userId) => {
  // Empêcher la suppression de certains acteurs
  if ( document.flags.mySystem?.protected ) {
    ui.notifications.error("Cet acteur est protégé et ne peut pas être supprimé.");
    return false;
  }
});

// deleteActor - APRÈS suppression
Hooks.on("deleteActor", (document, options, userId) => {
  // Nettoyage des références
  console.log(`Acteur ${document.name} supprimé`);
});
```

### Liste des Hooks par Document

| Document | preCreate | create | preUpdate | update | preDelete | delete |
|----------|-----------|--------|-----------|--------|-----------|--------|
| Actor | `preCreateActor` | `createActor` | `preUpdateActor` | `updateActor` | `preDeleteActor` | `deleteActor` |
| Item | `preCreateItem` | `createItem` | `preUpdateItem` | `updateItem` | `preDeleteItem` | `deleteItem` |
| ActiveEffect | `preCreateActiveEffect` | `createActiveEffect` | `preUpdateActiveEffect` | `updateActiveEffect` | `preDeleteActiveEffect` | `deleteActiveEffect` |
| ChatMessage | `preCreateChatMessage` | `createChatMessage` | `preUpdateChatMessage` | `updateChatMessage` | `preDeleteChatMessage` | `deleteChatMessage` |
| Combat | `preCreateCombat` | `createCombat` | `preUpdateCombat` | `updateCombat` | `preDeleteCombat` | `deleteCombat` |
| Scene | `preCreateScene` | `createScene` | `preUpdateScene` | `updateScene` | `preDeleteScene` | `deleteScene` |
| JournalEntry | `preCreateJournalEntry` | `createJournalEntry` | ... | ... | ... | ... |
| Macro | `preCreateMacro` | `createMacro` | ... | ... | ... | ... |
| Playlist | `preCreatePlaylist` | `createPlaylist` | ... | ... | ... | ... |

## Hooks de Rendu (render*)

Déclenchés lors du rendu des Applications.

```javascript
// Pattern: render[NomDeLApplication]
Hooks.on("renderActorSheet", (app, html, data) => {
  // app   = Instance de l'application
  // html  = Element DOM (jQuery ou HTMLElement selon la version)
  // data  = Données passées au template
  
  // Ajouter un bouton personnalisé
  const header = html.querySelector(".window-header");
  const button = document.createElement("button");
  button.innerHTML = '<i class="fas fa-dice"></i>';
  button.addEventListener("click", () => {
    app.document.rollInitiative();
  });
  header.appendChild(button);
});

// Autres hooks de rendu courants
Hooks.on("renderChatLog", (app, html, data) => {
  // Modifier le chat log
});

Hooks.on("renderChatMessage", (message, html, data) => {
  // Modifier un message de chat après rendu
});

Hooks.on("renderSettings", (app, html) => {
  // Ajouter des éléments au menu Settings
});

Hooks.on("renderCombatTracker", (app, html, data) => {
  // Modifier le tracker de combat
});

Hooks.on("renderCompendiumDirectory", (app, html) => {
  // Ajouter un bouton (comme le Compendium Browser)
});
```

## Hooks Personnalisés (dnd5e.*)

Le système dnd5e définit de nombreux hooks personnalisés pour permettre aux modules d'interagir avec ses fonctionnalités.

### Convention de Nommage

```
[nomDuSysteme].[pre?][action][Objet?]
```

- **Préfixe système** : `dnd5e.` pour éviter les collisions
- **pre** : Indique un hook qui peut bloquer l'action
- **action** : Verbe décrivant l'action (roll, use, apply, etc.)
- **objet** : Optionnel, précise le contexte

### Hooks de Jets de Dés

```javascript
// Avant un jet d'attaque
Hooks.on("dnd5e.preRollAttack", (config, dialog, message) => {
  // Modifier la configuration du jet
  config.parts.push("1d4"); // Ajouter un bonus
  
  // Bloquer le jet
  if ( someCondition ) return false;
});

// Après un jet d'attaque (V2 = nouvelle signature)
Hooks.on("dnd5e.rollAttackV2", (rolls, { subject, ammoUpdate }) => {
  // rolls = tableau de D20Roll
  // subject = l'activité qui a fait le jet
  console.log(`Attaque: ${rolls[0].total}`);
});

// Après affichage du message d'attaque
Hooks.on("dnd5e.postRollAttack", (rolls, { subject }) => {
  // Effets visuels, sons, etc.
});

// Hooks de dégâts
Hooks.on("dnd5e.rollDamage", (rolls, { subject }) => {
  console.log(`Dégâts totaux: ${rolls.reduce((t, r) => t + r.total, 0)}`);
});

// Hooks de jets de caractéristiques
Hooks.on("dnd5e.rollAbilityTest", (rolls, data) => {});
Hooks.on("dnd5e.rollAbilitySave", (rolls, data) => {});
Hooks.on("dnd5e.rollSkill", (rolls, data) => {});

// Jet de dé de vie
Hooks.on("dnd5e.preRollHitDie", (config, dialog, message) => {});
Hooks.on("dnd5e.rollHitDieV2", (rolls, { subject, updates }) => {
  // Peut bloquer avec return false
});
Hooks.on("dnd5e.postRollHitDie", (rolls, { subject }) => {});

// Jet de sauvegarde contre la mort
Hooks.on("dnd5e.rollDeathSave", (rolls, details) => {});
Hooks.on("dnd5e.postRollDeathSave", (rolls, { message, subject }) => {});

// Jet de concentration
Hooks.on("dnd5e.rollConcentration", (rolls, { subject }) => {});

// Jet d'initiative
Hooks.on("dnd5e.preRollInitiative", (actor, roll) => {
  // Modifier le jet d'initiative
});
Hooks.on("dnd5e.rollInitiative", (actor, combatants) => {});
```

### Hooks d'Utilisation d'Activités

```javascript
// Avant utilisation d'une activité (sort, capacité, etc.)
Hooks.on("dnd5e.preUseActivity", (activity, usageConfig, dialogConfig, messageConfig) => {
  // activity = l'activité utilisée
  // usageConfig = configuration d'utilisation
  // dialogConfig = configuration du dialogue
  // messageConfig = configuration du message de chat
  
  // Bloquer l'utilisation
  if ( !activity.item.system.uses.value ) return false;
});

// Après utilisation
Hooks.on("dnd5e.postUseActivity", (activity, usageConfig, results) => {
  // results = résultats de l'utilisation
});

// Cycle de consommation
Hooks.on("dnd5e.preActivityConsumption", (activity, usageConfig, messageConfig) => {});
Hooks.on("dnd5e.activityConsumption", (activity, usageConfig, messageConfig, updates) => {});
Hooks.on("dnd5e.postActivityConsumption", (activity, usageConfig, messageConfig, updates) => {});
```

### Hooks de Repos

```javascript
// Repos court
Hooks.on("dnd5e.preShortRest", (actor, config) => {
  // Peut bloquer avec return false
  if ( actor.system.attributes.exhaustion >= 6 ) {
    ui.notifications.warn("Trop épuisé pour se reposer.");
    return false;
  }
});

Hooks.on("dnd5e.shortRest", (actor, config) => {
  // Pendant le repos (avant finalisation)
});

// Repos long
Hooks.on("dnd5e.preLongRest", (actor, config) => {});
Hooks.on("dnd5e.longRest", (actor, config) => {});

// Fin de repos (court ou long)
Hooks.on("dnd5e.preRestCompleted", (actor, result, config) => {
  // Modifier les résultats avant application
});
Hooks.on("dnd5e.restCompleted", (actor, result, config) => {
  // Après que le repos a été complété
});

// Repos de groupe
Hooks.on("dnd5e.groupRestCompleted", (actor, results) => {});
```

### Hooks de Dégâts et Soins

```javascript
// Avant application des dégâts
Hooks.on("dnd5e.preApplyDamage", (actor, amount, updates, options) => {
  // Modifier les dégâts appliqués
  // Peut bloquer avec return false
});

// Après application des dégâts
Hooks.on("dnd5e.applyDamage", (actor, amount, options) => {
  // Notifications, effets visuels
});

// Calcul des dégâts (avec résistances/vulnérabilités)
Hooks.on("dnd5e.preCalculateDamage", (actor, damages, options) => {});
Hooks.on("dnd5e.calculateDamage", (actor, damages, options) => {});

// Événements de changement de PV
Hooks.on("dnd5e.damageActor", (actor, changes, data, userId) => {
  // Appelé quand un acteur subit des dégâts
});
Hooks.on("dnd5e.healActor", (actor, changes, data, userId) => {
  // Appelé quand un acteur est soigné
});
```

### Hooks de Concentration

```javascript
// Début de concentration
Hooks.on("dnd5e.preBeginConcentrating", (actor, item, effectData, activity) => {
  // Peut bloquer
});
Hooks.on("dnd5e.beginConcentrating", (actor, item, effect, activity) => {
  // Après que la concentration a commencé
});

// Fin de concentration
Hooks.on("dnd5e.preEndConcentration", (actor, effect) => {});
Hooks.on("dnd5e.endConcentration", (actor, effect) => {});
```

### Hooks de Transformation (Wild Shape, Polymorphe)

```javascript
// Transformation d'acteur
Hooks.on("dnd5e.transformActor", (original, target, data, settings, options) => {
  // Modifier les données de transformation
});

// Retour à la forme originale
Hooks.on("dnd5e.revertOriginalForm", (actor, options) => {});
```

### Hooks de Création d'Items

```javascript
// Création de parchemin à partir d'un sort
Hooks.on("dnd5e.preCreateScrollFromSpell", (itemData, options, config) => {});
Hooks.on("dnd5e.createScrollFromSpell", (spell, scrollData, config) => {});

// Initialisation de source d'item
Hooks.on("dnd5e.initializeItemSource", (item, data, options) => {});

// Affichage de carte d'item
Hooks.on("dnd5e.preDisplayCard", (item, messageConfig) => {});
Hooks.on("dnd5e.displayCard", (item, card) => {});
```

### Hooks d'Invocation

```javascript
// Invocation de créatures
Hooks.on("dnd5e.preSummon", (activity, profile, options) => {});
Hooks.on("dnd5e.preSummonToken", (activity, profile, tokenData, options) => {});
Hooks.on("dnd5e.summonToken", (activity, profile, tokenData, options) => {});
Hooks.on("dnd5e.postSummon", (activity, profile, tokens, options) => {});
```

### Hooks de Combat

```javascript
// Création de message de combat
Hooks.on("dnd5e.preCreateCombatMessage", (combatant, messageConfig) => {});

// Récupération pendant le combat
Hooks.on("dnd5e.preCombatRecovery", (combatant, periods) => {});
Hooks.on("dnd5e.combatRecovery", (combatant, periods, results) => {});
Hooks.on("dnd5e.postCombatRecovery", (combatant, periods, message) => {});
```

### Hooks de Messages de Chat

```javascript
// Rendu d'un message de chat
Hooks.on("dnd5e.renderChatMessage", (message, html) => {
  // Modifier le HTML du message après rendu
});

// Actions sur les messages de chat
Hooks.on("createChatMessage", dataModels.chatMessage.RequestMessageData.onCreateMessage);
Hooks.on("updateChatMessage", dataModels.chatMessage.RequestMessageData.onUpdateResultMessage);
```

### Hooks de Configuration

```javascript
// Configuration du calendrier
Hooks.on("dnd5e.setupCalendar", () => {
  // Retourner false pour empêcher la configuration par défaut
  return false;
});

// Configuration de l'initiative
Hooks.on("dnd5e.preConfigureInitiative", (actor, rollConfig) => {
  // Modifier la configuration d'initiative
});

// Configuration des slots de sorts
Hooks.on("dnd5e.prepareLeveledSlots", (spells, actor, progression) => {});
```

### Hooks d'Enchantement

```javascript
// Vérification d'enchantement
Hooks.on("dnd5e.canEnchant", (activity, item, errors) => {
  // Ajouter des erreurs pour bloquer l'enchantement
  if ( !item.system.price?.value ) {
    errors.push("L'item doit avoir une valeur pour être enchanté.");
  }
});
```

### Hooks de Menus Contextuels

```javascript
// Menu contextuel des activités
Hooks.on("dnd5e.getItemActivityContext", (activity, target, menuItems) => {
  // Ajouter des options au menu
  menuItems.push({
    name: "Mon action",
    icon: '<i class="fas fa-star"></i>',
    callback: () => doSomething(activity)
  });
});

// Menu contextuel des avancements
Hooks.on("dnd5e.getItemAdvancementContext", (advancement, target, menuItems) => {});

// Menus des répertoires
Hooks.on("getActorContextOptions", documents.Actor5e.addDirectoryContextOptions);
Hooks.on("getItemContextOptions", documents.Item5e.addDirectoryContextOptions);
```

## Créer ses Propres Hooks

### Déclaration et Appel

```javascript
// Dans votre système/module
const HOOKS = {
  // Hooks pouvant être bloqués (utilisent Hooks.call)
  PRE_CAST_SPELL: "mySystem.preCastSpell",
  PRE_APPLY_CONDITION: "mySystem.preApplyCondition",
  
  // Hooks informationnels (utilisent Hooks.callAll)
  CAST_SPELL: "mySystem.castSpell",
  APPLY_CONDITION: "mySystem.applyCondition",
  POST_LEVEL_UP: "mySystem.postLevelUp"
};

// Utilisation dans le code
async function castSpell(spell, options = {}) {
  // Hook "pre" - peut bloquer
  if ( Hooks.call(HOOKS.PRE_CAST_SPELL, this, spell, options) === false ) {
    return null;
  }
  
  // Logique principale
  const result = await this._performSpellCast(spell, options);
  
  // Hook "post" - informatif
  Hooks.callAll(HOOKS.CAST_SPELL, this, spell, result, options);
  
  return result;
}
```

### Best Practices pour les Hooks Personnalisés

```javascript
// 1. TOUJOURS préfixer avec le nom du système
// ✅ Bon
Hooks.callAll("dnd5e.rollAttack", ...);
Hooks.callAll("mySystem.castSpell", ...);

// ❌ Mauvais - risque de collision
Hooks.callAll("rollAttack", ...);
Hooks.callAll("castSpell", ...);

// 2. Documenter les paramètres avec JSDoc
/**
 * Un hook déclenché avant qu'un sort soit lancé.
 * @function mySystem.preCastSpell
 * @memberof hookEvents
 * @param {Actor} actor     L'acteur qui lance le sort
 * @param {Item} spell      Le sort lancé
 * @param {object} options  Options de lancement
 * @returns {boolean}       Retourner false pour bloquer
 */

// 3. Fournir des versions avec suffixe pour les changements d'API
Hooks.callAll("dnd5e.rollAttack", rolls, { subject, ammoUpdate });    // v1
Hooks.callAll("dnd5e.rollAttackV2", rolls, { subject, ammoUpdate });  // v2

// 4. Pattern pre/post cohérent
Hooks.call("mySystem.preAction", ...);   // Peut bloquer
Hooks.callAll("mySystem.action", ...);   // Principal
Hooks.callAll("mySystem.postAction", ...); // Nettoyage

// 5. Passer un contexte riche
Hooks.callAll("mySystem.levelUp", actor, {
  oldLevel: 4,
  newLevel: 5,
  classItem: classItem,
  choices: selectedChoices,
  updates: pendingUpdates
});
```

## Hooks Utilisés par dnd5e (Résumé)

Voici les principaux hooks enregistrés dans le fichier `dnd5e.mjs` :

### Hooks de Cycle de Vie (Hooks.once)

| Hook | Fichier | Description |
|------|---------|-------------|
| `init` | `dnd5e.mjs:57` | Configuration initiale du système |
| `i18nInit` | `dnd5e.mjs:482` | Configuration après chargement des traductions |
| `setup` | `dnd5e.mjs:432` | Configuration avancée après init de tous les modules |
| `ready` | `dnd5e.mjs:536` | Monde chargé, migrations, UI |

### Hooks d'Écoute (Hooks.on)

| Hook | Fichier | Description |
|------|---------|-------------|
| `hotbarDrop` | `dnd5e.mjs:538` | Création de macros depuis la hotbar |
| `renderGamePause` | `dnd5e.mjs:587` | Style de la pause de jeu |
| `renderSettings` | `dnd5e.mjs:599` | Modification du menu Settings |
| `applyCompendiumArt` | `dnd5e.mjs:605` | Application de l'art des compendiums |
| `renderChatPopout` | `dnd5e.mjs:607` | Rendu des popouts de chat |
| `getChatMessageContextOptions` | `dnd5e.mjs:608` | Options de contexte des messages |
| `renderChatLog` | `dnd5e.mjs:610` | Rendu du log de chat |
| `chatMessage` | `dnd5e.mjs:616` | Interception des messages de chat |
| `createChatMessage` | `dnd5e.mjs:617` | Création de messages |
| `updateChatMessage` | `dnd5e.mjs:618` | Mise à jour de messages |
| `renderActorDirectory` | `dnd5e.mjs:620` | Rendu du répertoire d'acteurs |
| `getActorContextOptions` | `dnd5e.mjs:622` | Options de contexte des acteurs |
| `getItemContextOptions` | `dnd5e.mjs:623` | Options de contexte des items |
| `renderCompendiumDirectory` | `dnd5e.mjs:625` | Rendu du répertoire de compendiums |
| `renderJournalEntryPageSheet` | `dnd5e.mjs:627` | Rendu des pages de journal |
| `renderActiveEffectConfig` | `dnd5e.mjs:629` | Configuration des effets actifs |
| `renderDocumentSheetConfig` | `dnd5e.mjs:631` | Configuration des feuilles |
| `targetToken` | `dnd5e.mjs:638` | Ciblage d'un token |
| `renderCombatTracker` | `dnd5e.mjs:640` | Rendu du tracker de combat |
| `preCreateScene` | `dnd5e.mjs:642` | Avant création de scène |
| `updateWorldTime` | `dnd5e.mjs:653` | Mise à jour du temps du monde |

## Debugging des Hooks

```javascript
// Activer le debug de tous les hooks
CONFIG.debug.hooks = true;

// Logger un hook spécifique
const originalCall = Hooks.call;
Hooks.call = function(name, ...args) {
  if ( name.startsWith("dnd5e.") ) {
    console.log(`Hook called: ${name}`, args);
  }
  return originalCall.call(this, name, ...args);
};

// Lister tous les listeners d'un hook
console.log(Hooks.events["updateActor"]);
```

## Liens vers la Documentation Officielle

- [Hooks API](https://foundryvtt.com/api/classes/Hooks.html)
- [Hook Events](https://foundryvtt.com/api/hookEvents.html)
- [Document Lifecycle](https://foundryvtt.com/article/documents/)

---

*Documentation précédente : [04-applications.md](./04-applications.md) - Feuilles et Applications (AppV2)*

*Documentation suivante : [06-dice.md](./06-dice.md) - Système de Dés Personnalisés*
