# Applications et Sheets

> Documentation technique pour FoundryVTT v13

## Vue d'ensemble

Les **Applications** sont les fenêtres et interfaces utilisateur dans FoundryVTT. Avec la version 13, Foundry introduit **ApplicationV2**, une refonte complète de l'API d'interface qui remplace l'ancienne `Application` v1.

Les **Sheets** (feuilles) sont des applications spécialisées pour afficher et éditer les Documents (Actors, Items, etc.).

```
foundry.applications.api.ApplicationV2
    │
    ├── HandlebarsApplicationMixin
    │   └── ApplicationV2Mixin (dnd5e)
    │       └── Application5e
    │
    └── foundry.applications.api.DocumentSheetV2
        └── foundry.applications.sheets.ActorSheetV2
        │   └── BaseActorSheet (dnd5e)
        │       ├── CharacterActorSheet
        │       ├── NPCActorSheet
        │       └── VehicleActorSheet
        │
        └── foundry.applications.sheets.ItemSheetV2
            └── ItemSheet5e (dnd5e)
```

## Application v1 vs ApplicationV2

### Différences Majeures

| Aspect | Application v1 (Dépréciée) | ApplicationV2 (Moderne v13) |
|--------|---------------------------|----------------------------|
| **Namespace** | `Application`, `FormApplication` | `foundry.applications.api.ApplicationV2` |
| **Rendu** | Méthode `getData()` + `activateListeners()` | `_prepareContext()` + `_onRender()` |
| **Templates** | Un template unique | Système de **PARTS** (multiples templates) |
| **Options** | `static get defaultOptions()` | `static DEFAULT_OPTIONS` |
| **Actions** | Event listeners jQuery manuels | Système d'actions déclaratif `data-action` |
| **jQuery** | Dépendance forte | Sans jQuery natif |
| **Réactivité** | Re-rendu complet | Re-rendu partiel par PART |

### Migration v1 → v2

```javascript
// ❌ Application v1 (ancienne API)
class OldSheet extends Application {
  static get defaultOptions() {
    return foundry.utils.mergeObject(super.defaultOptions, {
      template: "path/to/template.hbs",
      width: 500,
      height: 400
    });
  }
  
  getData() {
    return { data: this.object };
  }
  
  activateListeners(html) {
    super.activateListeners(html);
    html.find(".my-button").click(this._onButtonClick.bind(this));
  }
}

// ✅ ApplicationV2 (nouvelle API v13)
class ModernSheet extends foundry.applications.api.ApplicationV2 {
  static DEFAULT_OPTIONS = {
    position: { width: 500, height: 400 },
    actions: {
      myAction: ModernSheet.#onMyAction
    }
  };
  
  static PARTS = {
    main: { template: "path/to/template.hbs" }
  };
  
  async _prepareContext(options) {
    return { data: this.document };
  }
  
  static #onMyAction(event, target) {
    // Gestionnaire d'action
  }
}
```

---

## ApplicationV2 - API Moderne

### Structure de Classe

```javascript
const { ApplicationV2, HandlebarsApplicationMixin } = foundry.applications.api;

class MyApplication extends HandlebarsApplicationMixin(ApplicationV2) {
  
  /** @override */
  static DEFAULT_OPTIONS = {
    // Identifiant unique de la classe
    id: "my-app-{id}",
    
    // Classes CSS appliquées à l'application
    classes: ["my-system", "my-app"],
    
    // Tag HTML de l'application (default: "div")
    tag: "div",
    
    // Configuration de la fenêtre
    window: {
      title: "My Application",
      icon: "fas fa-cog",
      resizable: true,
      minimizable: true,
      controls: []  // Boutons de contrôle dans l'en-tête
    },
    
    // Position et dimensions
    position: {
      width: 600,
      height: 400,
      top: null,
      left: null
    },
    
    // Actions déclaratives (remplace activateListeners)
    actions: {
      save: MyApplication.#onSave,
      delete: MyApplication.#onDelete,
      toggle: { 
        handler: MyApplication.#onToggle,
        buttons: [0, 2]  // Boutons souris acceptés
      }
    },
    
    // Configuration du formulaire (pour les feuilles)
    form: {
      handler: MyApplication.#onSubmit,
      submitOnChange: true,
      closeOnSubmit: false
    }
  };
  
  /** @override */
  static PARTS = {
    header: {
      template: "path/to/header.hbs"
    },
    content: {
      template: "path/to/content.hbs",
      scrollable: [""],  // Sélecteurs des zones scrollables
      templates: [       // Templates additionnels à précharger
        "path/to/partial1.hbs",
        "path/to/partial2.hbs"
      ]
    },
    footer: {
      template: "templates/generic/form-footer.hbs"
    }
  };
}
```

### Cycle de Vie du Rendu

```
┌─────────────────────────────────────────────────────────────────┐
│                    render(options)                               │
└─────────────────────────────────────────────────────────────────┘
                              │
                              ▼
┌─────────────────────────────────────────────────────────────────┐
│              _configureRenderOptions(options)                    │
│  - Configure les options de rendu                                │
│  - Définit quelles PARTS doivent être rendues                    │
└─────────────────────────────────────────────────────────────────┘
                              │
                              ▼
┌─────────────────────────────────────────────────────────────────┐
│                    _prepareContext(options)                      │
│  - Prépare le contexte de données pour les templates             │
│  - Retourne un objet passé à tous les templates                  │
└─────────────────────────────────────────────────────────────────┘
                              │
                              ▼
┌─────────────────────────────────────────────────────────────────┐
│            _preparePartContext(partId, context, options)         │
│  - Enrichit le contexte pour chaque PART spécifique              │
│  - Appelé une fois par PART à rendre                             │
└─────────────────────────────────────────────────────────────────┘
                              │
                              ▼
┌─────────────────────────────────────────────────────────────────┐
│                _renderHTML(context, options)                     │
│  - Compile les templates avec le contexte                        │
│  - Génère le HTML pour chaque PART                               │
└─────────────────────────────────────────────────────────────────┘
                              │
                    ┌─────────┴─────────┐
            Premier rendu?          Re-rendu?
                    │                   │
                    ▼                   ▼
┌───────────────────────────┐ ┌───────────────────────────┐
│   _renderFrame(options)   │ │  _replaceHTML(result...)  │
│   Crée le conteneur       │ │  Remplace uniquement les  │
└───────────────────────────┘ │  PARTS modifiées          │
            │                 └───────────────────────────┘
            ▼                           │
┌───────────────────────────┐           │
│  _onFirstRender(context)  │           │
│  Initialisation unique    │           │
└───────────────────────────┘           │
            │                           │
            └───────────┬───────────────┘
                        ▼
┌─────────────────────────────────────────────────────────────────┐
│                 _onRender(context, options)                      │
│  - Appelé après chaque rendu                                     │
│  - Initialiser les écouteurs d'événements personnalisés          │
│  - Configurer les composants UI                                  │
└─────────────────────────────────────────────────────────────────┘
```

### Préparation du Contexte

```javascript
class MySheet extends ApplicationV2 {
  
  /**
   * Prépare le contexte global partagé par tous les templates.
   * @param {object} options - Options de rendu
   * @returns {Promise<object>} - Contexte pour les templates
   */
  async _prepareContext(options) {
    const context = await super._prepareContext(options);
    
    // Données de base
    context.document = this.document;
    context.system = this.document.system;
    context.source = this.document.system._source; // Données non-dérivées
    
    // Données d'édition
    context.editable = this.isEditable;
    context.owner = this.document.isOwner;
    
    // Configuration système
    context.CONFIG = CONFIG.DND5E;
    
    // Données enrichies (HTML)
    context.enriched = {
      description: await TextEditor.enrichHTML(
        this.document.system.description,
        { secrets: this.document.isOwner, relativeTo: this.document }
      )
    };
    
    return context;
  }
  
  /**
   * Prépare le contexte spécifique à chaque PART.
   * @param {string} partId - Identifiant de la PART
   * @param {object} context - Contexte préparé par _prepareContext
   * @param {object} options - Options de rendu
   */
  async _preparePartContext(partId, context, options) {
    context = await super._preparePartContext(partId, context, options);
    
    switch (partId) {
      case "header":
        context.title = this.document.name;
        break;
      case "details":
        context.fields = this.document.system.schema.fields;
        context.abilities = this._prepareAbilities();
        break;
      case "inventory":
        context.items = this._prepareItems();
        break;
    }
    
    return context;
  }
}
```

### Système de PARTS

Les PARTS permettent un rendu modulaire et performant :

```javascript
static PARTS = {
  // PART simple avec un template
  header: {
    template: "systems/dnd5e/templates/actors/character-header.hbs"
  },
  
  // PART avec templates additionnels (partials)
  inventory: {
    template: "systems/dnd5e/templates/actors/tabs/character-inventory.hbs",
    templates: [
      "systems/dnd5e/templates/inventory/inventory.hbs",
      "systems/dnd5e/templates/inventory/encumbrance.hbs"
    ],
    scrollable: [""]  // La PART entière est scrollable
  },
  
  // PART avec conteneur
  sidebar: {
    container: { 
      id: "main",
      classes: ["main-content"] 
    },
    template: "systems/dnd5e/templates/actors/character-sidebar.hbs"
  },
  
  // PART conditionnelle (gérée dans _configureRenderParts)
  bastion: {
    container: { classes: ["tab-body"], id: "tabs" },
    template: "systems/dnd5e/templates/actors/tabs/character-bastion.hbs",
    scrollable: [""]
  }
};

/** @inheritDoc */
_configureRenderParts(options) {
  const parts = super._configureRenderParts(options);
  
  // Retirer une PART conditionnellement
  if (!this.shouldShowBastion()) {
    delete parts.bastion;
  }
  
  return parts;
}
```

### Système d'Actions

Le système d'actions remplace `activateListeners()` avec une approche déclarative :

```html
<!-- Template HTML -->
<button type="button" data-action="save">Sauvegarder</button>
<button type="button" data-action="delete" data-confirm="true">Supprimer</button>
<div data-action="toggle" data-item-id="{{item.id}}">Toggle</div>
```

```javascript
static DEFAULT_OPTIONS = {
  actions: {
    // Action simple
    save: MySheet.#onSave,
    
    // Action avec configuration
    delete: {
      handler: MySheet.#onDelete,
      buttons: [0]  // Clic gauche uniquement
    },
    
    // Action héritée d'un mixin
    toggleCollapsed: BaseApplication5e.#toggleCollapsed
  }
};

/**
 * Gestionnaire d'action.
 * @this {MySheet} - Instance de l'application (lié automatiquement)
 * @param {Event} event - L'événement déclencheur
 * @param {HTMLElement} target - L'élément avec data-action
 */
static #onSave(event, target) {
  // Récupérer des données depuis l'élément
  const itemId = target.dataset.itemId;
  const item = this.document.items.get(itemId);
  
  // Soumettre le formulaire avec des données supplémentaires
  this.submit({ updateData: { saved: true } });
}

static async #onDelete(event, target) {
  const confirmed = await Dialog.confirm({
    title: "Confirmer",
    content: "Supprimer cet élément ?"
  });
  if (confirmed) this.document.delete();
}
```

---

## DocumentSheetV2

`DocumentSheetV2` étend `ApplicationV2` avec des fonctionnalités spécifiques aux Documents :

### Structure de Base

```javascript
const { DocumentSheetV2 } = foundry.applications.api;

class MyDocumentSheet extends HandlebarsApplicationMixin(DocumentSheetV2) {
  
  /** @override */
  static DEFAULT_OPTIONS = {
    classes: ["my-system", "document-sheet"],
    form: {
      submitOnChange: true,  // Soumet le formulaire à chaque modification
      closeOnSubmit: false
    },
    window: {
      resizable: true
    }
  };
  
  // Le document géré par la feuille
  get document() { return this.object; }
}
```

### Binding Automatique des Données

DocumentSheetV2 supporte le binding automatique via les attributs `name` :

```html
<!-- Input standard lié à document.name -->
<input type="text" name="name" value="{{document.name}}">

<!-- Input lié à document.system.health.value -->
<input type="number" name="system.health.value" value="{{system.health.value}}">

<!-- Checkbox liée à document.system.active -->
<input type="checkbox" name="system.active" {{checked system.active}}>

<!-- Select lié à document.system.type -->
<select name="system.type">
  {{selectOptions typeOptions selected=system.type}}
</select>

<!-- Champ du DataModel avec validation -->
{{formInput fields.description value=system.description}}
```

### Gestion des Formulaires

```javascript
class MySheet extends DocumentSheetV2 {
  
  /**
   * Appelé à la soumission du formulaire.
   * @param {Event} event - L'événement de soumission
   * @param {HTMLFormElement} form - Le formulaire
   * @param {FormDataExtended} formData - Données du formulaire parsées
   */
  async _processFormData(event, form, formData) {
    const submitData = super._processFormData(event, form, formData);
    
    // Transformer les données avant soumission
    if (submitData.system?.properties) {
      submitData.system.properties = Object.keys(submitData.system.properties)
        .filter(k => submitData.system.properties[k]);
    }
    
    return submitData;
  }
  
  /**
   * Handler de soumission personnalisé.
   */
  static async _onSubmit(event, form, formData) {
    const submitData = this._processFormData(event, form, formData);
    await this.document.update(submitData);
  }
}
```

---

## Sheets Spécifiques - ActorSheetV2 & ItemSheetV2

### ActorSheetV2

```javascript
// Extension de ActorSheetV2 dans dnd5e
class BaseActorSheet extends PrimarySheetMixin(
  ApplicationV2Mixin(foundry.applications.sheets.ActorSheetV2)
) {
  
  static DEFAULT_OPTIONS = {
    classes: ["actor", "standard-form"],
    elements: {
      effects: "dnd5e-effects",    // Composant custom
      inventory: "dnd5e-inventory"
    },
    form: {
      submitOnChange: true
    },
    window: {
      resizable: true,
      controls: [
        {
          action: "restoreTransformation",
          icon: "fa-solid fa-backward",
          label: "DND5E.TRANSFORM.Action.Restore",
          ownership: "OWNER",
          visible: BaseActorSheet.#canRestoreTransformation
        }
      ]
    }
  };
  
  static PARTS = {
    header: {
      template: "systems/dnd5e/templates/actors/character-header.hbs"
    },
    sidebar: {
      container: { classes: ["main-content"], id: "main" },
      template: "systems/dnd5e/templates/actors/character-sidebar.hbs"
    },
    details: {
      container: { classes: ["tab-body"], id: "tabs" },
      template: "systems/dnd5e/templates/actors/tabs/character-details.hbs",
      scrollable: [""]
    },
    inventory: {
      container: { classes: ["tab-body"], id: "tabs" },
      template: "systems/dnd5e/templates/actors/tabs/character-inventory.hbs",
      templates: [
        "systems/dnd5e/templates/inventory/inventory.hbs",
        "systems/dnd5e/templates/inventory/encumbrance.hbs"
      ],
      scrollable: [""]
    }
  };
  
  // Onglets de la feuille
  static TABS = [
    { tab: "details", label: "DND5E.Details", icon: "fas fa-cog" },
    { tab: "inventory", label: "DND5E.Inventory", svg: "systems/dnd5e/icons/svg/backpack.svg" },
    { tab: "features", label: "DND5E.Features", icon: "fas fa-list" },
    { tab: "spells", label: "TYPES.Item.spellPl", icon: "fas fa-book" }
  ];
}
```

### Sélection de Sheet par Type

Foundry permet d'associer différentes sheets à différents types d'acteurs :

```javascript
// Dans dnd5e.mjs (hook init)
const DocumentSheetConfig = foundry.applications.apps.DocumentSheetConfig;

// Sheet pour les personnages joueurs
DocumentSheetConfig.registerSheet(Actor, "dnd5e", CharacterActorSheet, {
  types: ["character"],  // Type d'acteur
  makeDefault: true,
  label: "DND5E.SheetClass.Character"
});

// Sheet pour les PNJ
DocumentSheetConfig.registerSheet(Actor, "dnd5e", NPCActorSheet, {
  types: ["npc"],
  makeDefault: true,
  label: "DND5E.SheetClass.NPC"
});

// Sheet pour les véhicules
DocumentSheetConfig.registerSheet(Actor, "dnd5e", VehicleActorSheet, {
  types: ["vehicle"],
  makeDefault: true,
  label: "DND5E.SheetClass.Vehicle"
});

// Sheet pour les groupes
DocumentSheetConfig.registerSheet(Actor, "dnd5e", GroupActorSheet, {
  types: ["group"],
  makeDefault: true,
  label: "DND5E.SheetClass.Group"
});
```

### ItemSheetV2

```javascript
class ItemSheet5e extends PrimarySheetMixin(DocumentSheet5e) {
  
  static DEFAULT_OPTIONS = {
    classes: ["item"],
    elements: {
      activities: "dnd5e-activities",
      effects: "dnd5e-effects"
    },
    form: {
      submitOnChange: true
    },
    position: {
      width: 500
    },
    window: {
      resizable: false
    }
  };
  
  static PARTS = {
    header: {
      template: "systems/dnd5e/templates/items/header.hbs"
    },
    tabs: {
      template: "systems/dnd5e/templates/shared/horizontal-tabs.hbs"
    },
    description: {
      template: "systems/dnd5e/templates/items/description.hbs",
      scrollable: [""]
    },
    details: {
      template: "systems/dnd5e/templates/items/details.hbs",
      scrollable: [""]
    },
    activities: {
      template: "systems/dnd5e/templates/items/activities.hbs",
      scrollable: [""]
    },
    effects: {
      template: "systems/dnd5e/templates/items/effects.hbs",
      scrollable: [""]
    }
  };
  
  // Onglets conditionnels
  static TABS = [
    { tab: "description", label: "DND5E.ITEM.SECTIONS.Description" },
    { tab: "details", label: "DND5E.ITEM.SECTIONS.Details", 
      condition: this.isItemIdentified.bind(this) },
    { tab: "activities", label: "DND5E.ITEM.SECTIONS.Activities", 
      condition: this.itemHasActivities.bind(this) },
    { tab: "effects", label: "DND5E.ITEM.SECTIONS.Effects", 
      condition: this.itemHasEffects.bind(this) }
  ];
  
  // Vérifie si l'item est identifié (pour masquer certains onglets)
  static isItemIdentified(item) {
    return game.user.isGM || (item.system.identified !== false);
  }
}
```

---

## Composants UI Personnalisés

### Custom Elements

dnd5e utilise des Web Components personnalisés :

```javascript
// Définition d'un Custom Element
class EffectsElement extends HTMLElement {
  static tagName = "dnd5e-effects";
  
  connectedCallback() {
    this.#app = foundry.applications.instances.get(
      this.closest(".application")?.id
    );
    
    // Initialiser les listeners
    for (const control of this.querySelectorAll("[data-action]")) {
      control.addEventListener("click", event => {
        this._onAction(event.currentTarget, event.currentTarget.dataset.action);
      });
    }
    
    // Menu contextuel
    new ContextMenu5e(this, "[data-effect-id]", [], {
      onOpen: element => {
        const effect = this.getEffect(element.dataset);
        ui.context.menuItems = this._getContextOptions(effect);
      },
      jQuery: false
    });
  }
  
  get document() {
    return this.app.document;
  }
}

// Enregistrement
customElements.define(EffectsElement.tagName, EffectsElement);
```

### Composants Disponibles dans dnd5e

| Composant | Tag HTML | Rôle |
|-----------|----------|------|
| `EffectsElement` | `<dnd5e-effects>` | Liste des effets actifs |
| `InventoryElement` | `<dnd5e-inventory>` | Gestion de l'inventaire |
| `ActivitiesElement` | `<dnd5e-activities>` | Liste des activités |
| `CheckboxElement` | `<dnd5e-checkbox>` | Checkbox stylisée |
| `SlideToggle` | `<slide-toggle>` | Toggle animé |
| `IconElement` | `<dnd5e-icon>` | Icône SVG |
| `FiligreeBox` | `<filigree-box>` | Conteneur décoratif |

### Dialogs

```javascript
// Dialog simple avec Dialog5e
class MyDialog extends Dialog5e {
  static DEFAULT_OPTIONS = {
    templates: ["path/to/content.hbs"],
    window: {
      title: "Mon Dialog"
    }
  };
  
  static PARTS = {
    content: {
      template: "systems/dnd5e/templates/shared/dialog-content.hbs"
    },
    footer: {
      template: "templates/generic/form-footer.hbs"
    }
  };
  
  async _prepareContentContext(context, options) {
    context.content = await TextEditor.enrichHTML(this.options.content);
    return context;
  }
  
  async _prepareFooterContext(context, options) {
    context.buttons = [
      { type: "submit", icon: "fas fa-check", label: "Confirmer" },
      { type: "button", icon: "fas fa-times", label: "Annuler", 
        action: "close" }
    ];
    return context;
  }
}

// Utilisation
const dialog = new MyDialog({ content: "<p>Contenu</p>" });
dialog.render({ force: true });
```

### Context Menu

```javascript
// Menu contextuel personnalisé
class ContextMenu5e extends foundry.applications.ux.ContextMenu {
  /** @override */
  _setPosition(html, target, options = {}) {
    html.classList.add("dnd5e2");
    return this._setFixedPosition(html, target, options);
  }
  
  // Déclencher un menu contextuel programmatiquement
  static triggerEvent(event) {
    event.preventDefault();
    event.stopPropagation();
    const { clientX, clientY } = event;
    const selector = "[data-id],[data-effect-id],[data-item-id]";
    const target = event.target.closest(selector);
    target?.dispatchEvent(new PointerEvent("contextmenu", {
      view: window, bubbles: true, cancelable: true, clientX, clientY
    }));
  }
}

// Usage dans un template
<button data-context-menu data-action="showContextMenu">
  <i class="fas fa-ellipsis-v"></i>
</button>
```

---

## Enregistrement des Sheets

### DocumentSheetConfig API

```javascript
const DocumentSheetConfig = foundry.applications.apps.DocumentSheetConfig;

// Désenregistrer la sheet par défaut de Foundry
DocumentSheetConfig.unregisterSheet(Actor, "core", foundry.appv1.sheets.ActorSheet);

// Enregistrer une nouvelle sheet
DocumentSheetConfig.registerSheet(Actor, "mon-systeme", MaActorSheet, {
  types: ["character", "npc"],  // Types d'acteurs concernés
  makeDefault: true,            // Sheet par défaut
  label: "MON_SYSTEME.SheetName" // Clé de traduction pour le label
});

// Enregistrer une sheet spécifique à un type
DocumentSheetConfig.registerSheet(Item, "mon-systeme", MaWeaponSheet, {
  types: ["weapon"],
  makeDefault: true,
  label: "MON_SYSTEME.WeaponSheet"
});

// Sheet alternative (non par défaut)
DocumentSheetConfig.registerSheet(Actor, "mon-systeme", MaActorSheetAlternative, {
  types: ["character"],
  makeDefault: false,
  label: "MON_SYSTEME.AlternativeSheet"
});
```

### Configuration Globale

```javascript
// Les sheets enregistrées sont stockées dans CONFIG
CONFIG.Actor.sheetClasses; // Map des sheets par type
CONFIG.Item.sheetClasses;

// Structure de CONFIG.Actor.sheetClasses
{
  "character": {
    "mon-systeme.MaActorSheet": {
      id: "mon-systeme.MaActorSheet",
      cls: MaActorSheet,
      default: true,
      label: "Ma Feuille de Personnage"
    }
  },
  "npc": {
    "mon-systeme.MaNPCSheet": { ... }
  }
}
```

### Prototype Token Config

```javascript
// Sheet personnalisée pour la configuration des tokens
CONFIG.Token.prototypeSheetClass = MyPrototypeTokenConfig;

DocumentSheetConfig.unregisterSheet(
  TokenDocument, 
  "core", 
  foundry.applications.sheets.TokenConfig
);

DocumentSheetConfig.registerSheet(TokenDocument, "mon-systeme", MyTokenConfig, {
  label: "MON_SYSTEME.TokenConfig"
});
```

---

## Patterns Avancés

### Mixin Pattern (dnd5e)

dnd5e utilise des mixins pour partager des fonctionnalités :

```javascript
// Mixin pour ApplicationV2
export default function ApplicationV2Mixin(Base) {
  class BaseApplication5e extends HandlebarsApplicationMixin(Base) {
    
    static DEFAULT_OPTIONS = {
      actions: {
        toggleCollapsed: BaseApplication5e.#toggleCollapsed
      },
      classes: ["dnd5e2"]
    };
    
    // Sections collapsibles persistantes
    #expandedSections = new Map();
    get expandedSections() { return this.#expandedSections; }
    
    /** @inheritDoc */
    async _prepareContext(options) {
      const context = await super._prepareContext(options);
      context.CONFIG = CONFIG.DND5E;
      return context;
    }
    
    static #toggleCollapsed(event, target) {
      target.classList.toggle("collapsed");
      this.#expandedSections.set(
        target.closest("[data-expand-id]")?.dataset.expandId,
        !target.classList.contains("collapsed")
      );
    }
  }
  
  return BaseApplication5e;
}

// Usage
class Application5e extends ApplicationV2Mixin(ApplicationV2) {}
class DocumentSheet5e extends ApplicationV2Mixin(DocumentSheetV2) {}
```

### PrimarySheetMixin

Pour les sheets principales (Actors, Items) :

```javascript
export default function PrimarySheetMixin(Base) {
  return class PrimarySheet5e extends DragDropApplicationMixin(Base) {
    
    // Modes d'édition
    static MODES = { PLAY: 1, EDIT: 2 };
    _mode = null;
    
    static TABS = [];
    
    /** @inheritDoc */
    async _prepareContext(options) {
      const context = await super._prepareContext(options);
      context.owner = this.document.isOwner;
      context.locked = !this.isEditable;
      context.editable = this.isEditable && (this._mode === this.constructor.MODES.EDIT);
      context.tabs = this._getTabs();
      return context;
    }
    
    _getTabs() {
      return this.constructor.TABS.reduce((tabs, { tab, condition, ...config }) => {
        if (!condition || condition(this.document)) {
          tabs[tab] = {
            ...config,
            id: tab,
            group: "primary",
            active: this.tabGroups.primary === tab
          };
        }
        return tabs;
      }, {});
    }
    
    // Toggle mode édition
    async _onChangeSheetMode(event) {
      const toggle = event.currentTarget;
      this._mode = toggle.checked ? this.constructor.MODES.EDIT : this.constructor.MODES.PLAY;
      await this.submit();
      this.render();
    }
  };
}
```

### Drag & Drop

```javascript
class MySheet extends PrimarySheetMixin(DocumentSheet5e) {
  
  /** @inheritDoc */
  async _onDragStart(event) {
    const li = event.currentTarget;
    let dragData;
    
    // Item drag
    if (li.dataset.itemId) {
      const item = this.actor.items.get(li.dataset.itemId);
      dragData = item.toDragData();
    }
    // Effect drag
    else if (li.dataset.effectId) {
      const effect = this.actor.effects.get(li.dataset.effectId);
      dragData = effect.toDragData();
    }
    
    if (dragData) {
      event.dataTransfer.setData("text/plain", JSON.stringify(dragData));
    }
  }
  
  /** @inheritDoc */
  async _onDrop(event) {
    const data = TextEditor.getDragEventData(event);
    
    switch (data.type) {
      case "Item":
        return this._onDropItem(event, data);
      case "ActiveEffect":
        return this._onDropActiveEffect(event, data);
      case "Actor":
        return this._onDropActor(event, data);
    }
  }
  
  async _onDropItem(event, data) {
    const item = await Item.implementation.fromDropData(data);
    // Créer une copie de l'item sur cet acteur
    return Item.create(item.toObject(), { parent: this.actor });
  }
}
```

---

## Bonnes Pratiques

### 1. Utiliser les PARTS pour le rendu partiel

```javascript
// ✅ Bon : Rendre uniquement les parties modifiées
async render(options = {}) {
  options.parts = ["inventory"]; // Ne re-rend que l'inventaire
  return super.render(options);
}
```

### 2. Séparer préparation et rendu

```javascript
// ✅ Bon : Logique dans _prepareContext
async _prepareContext(options) {
  return {
    items: this._prepareItems(),        // Méthode dédiée
    abilities: this._prepareAbilities() // Méthode dédiée
  };
}

// Méthodes de préparation réutilisables
_prepareItems() {
  return Array.from(this.actor.items)
    .filter(i => i.type !== "container")
    .sort((a, b) => a.sort - b.sort);
}
```

### 3. Actions statiques avec `this` lié

```javascript
// ✅ Bon : Méthode statique privée avec this lié automatiquement
static DEFAULT_OPTIONS = {
  actions: { myAction: MySheet.#myAction }
};

static #myAction(event, target) {
  // `this` est l'instance de MySheet
  this.document.update({ name: "Nouveau nom" });
}
```

### 4. Éviter les manipulations DOM excessives

```javascript
// ❌ Mauvais : Manipulation DOM dans _onRender
_onRender(context, options) {
  this.element.querySelector(".title").textContent = "Titre";
}

// ✅ Bon : Passer les données via le contexte
async _prepareContext(options) {
  return { title: "Titre" };
}
// Template: <h1>{{title}}</h1>
```

---

## Liens API Officiels

- [ApplicationV2](https://foundryvtt.com/api/classes/foundry.applications.api.ApplicationV2.html)
- [DocumentSheetV2](https://foundryvtt.com/api/classes/foundry.applications.api.DocumentSheetV2.html)
- [ActorSheetV2](https://foundryvtt.com/api/classes/foundry.applications.sheets.ActorSheetV2.html)
- [ItemSheetV2](https://foundryvtt.com/api/classes/foundry.applications.sheets.ItemSheetV2.html)
- [HandlebarsApplicationMixin](https://foundryvtt.com/api/functions/foundry.applications.api.HandlebarsApplicationMixin.html)
- [DocumentSheetConfig](https://foundryvtt.com/api/classes/foundry.applications.apps.DocumentSheetConfig.html)

---

*Documentation précédente : [03-documents.md](./03-documents.md) - Documents et Extensions*

*Documentation suivante : [05-dice.md](./05-dice.md) - Système de Dés et Rolls (à venir)*
