# Documents et Extensions

> Documentation technique pour FoundryVTT v13

## Vue d'ensemble

Les **Documents** sont les entités fondamentales de FoundryVTT. Ils représentent les données persistantes du monde : acteurs, objets, scènes, etc. 

Étendre les classes de Documents permet d'ajouter :
- Des méthodes métier personnalisées
- Une logique de préparation des données
- Des hooks de cycle de vie (création, modification, suppression)
- Des comportements spécifiques au système

## Hiérarchie des Documents

```
foundry.abstract.Document
    │
    ├── Actor
    │   └── Actor5e (extension système)
    │
    ├── Item  
    │   └── Item5e
    │
    ├── ActiveEffect
    │   └── ActiveEffect5e
    │
    ├── ChatMessage
    │   └── ChatMessage5e
    │
    ├── Combat
    │   └── Combat5e
    │
    ├── Combatant
    │   └── Combatant5e
    │
    ├── JournalEntry / JournalEntryPage
    ├── Scene / Token
    ├── User
    └── ... (autres documents core)
```

## Documents Principaux à Étendre

| Document | Usage | Extension Commune |
|----------|-------|-------------------|
| `Actor` | Personnages, PNJ, créatures | Calculs de stats, jets, repos |
| `Item` | Objets, sorts, capacités | Utilisation, dégâts, propriétés |
| `ActiveEffect` | Buffs, debuffs, conditions | Transfert, expiration |
| `ChatMessage` | Messages de chat | Formatage, boutons d'action |
| `Combat` | Rencontres | Initiative, tours personnalisés |
| `Combatant` | Participants au combat | Statut, initiatives spéciales |
| `TokenDocument` | Représentation sur canvas | Bars, overlays |

## Cycle de Vie des Documents

```
┌─────────────────────────────────────────────────────────────────┐
│                    CRÉATION (_preCreate → _onCreate)            │
├─────────────────────────────────────────────────────────────────┤
│  _preCreate(data, options, user)                                │
│    ├── Valider/modifier les données AVANT création             │
│    ├── Peut annuler la création en retournant false             │
│    └── Appelé côté client ET serveur                            │
│                                                                 │
│  _onCreate(data, options, userId)                               │
│    ├── Réagir APRÈS création (effets secondaires)               │
│    └── Données déjà persistées en base                          │
└─────────────────────────────────────────────────────────────────┘
                              │
                              ▼
┌─────────────────────────────────────────────────────────────────┐
│                    LECTURE (prepareData)                        │
├─────────────────────────────────────────────────────────────────┤
│  prepareData()                                                  │
│    ├── prepareBaseData()       // Données de base               │
│    ├── prepareEmbeddedDocuments() // Items, Effects             │
│    ├── applyActiveEffects()    // Appliquer les effets          │
│    └── prepareDerivedData()    // Données calculées             │
└─────────────────────────────────────────────────────────────────┘
                              │
                              ▼
┌─────────────────────────────────────────────────────────────────┐
│                    MODIFICATION (_preUpdate → _onUpdate)        │
├─────────────────────────────────────────────────────────────────┤
│  _preUpdate(changed, options, user)                             │
│    ├── Modifier/valider les changements                         │
│    └── Peut annuler en retournant false                         │
│                                                                 │
│  _onUpdate(changed, options, userId)                            │
│    ├── Réagir aux changements                                   │
│    └── Re-render les sheets si nécessaire                       │
└─────────────────────────────────────────────────────────────────┘
                              │
                              ▼
┌─────────────────────────────────────────────────────────────────┐
│                    SUPPRESSION (_preDelete → _onDelete)         │
├─────────────────────────────────────────────────────────────────┤
│  _preDelete(options, user)                                      │
│    └── Peut annuler en retournant false                         │
│                                                                 │
│  _onDelete(options, userId)                                     │
│    └── Nettoyage, effets secondaires                            │
└─────────────────────────────────────────────────────────────────┘
```

## Extension de Actor

### Structure de Base

```javascript
// module/documents/actor.mjs
import SystemDocumentMixin from "./mixins/document.mjs";

export default class Actor5e extends SystemDocumentMixin(Actor) {
  
  /** Icône par défaut pour les nouveaux acteurs */
  static DEFAULT_ICON = "systems/dnd5e/icons/svg/documents/actor.svg";
  
  /* -------------------------------------------- */
  /*  Propriétés                                  */
  /* -------------------------------------------- */
  
  /**
   * Cache des classes du personnage
   * @type {Record<string, Item5e>}
   */
  get classes() {
    return Object.fromEntries(
      this.itemTypes.class.map(cls => [cls.identifier, cls])
    );
  }
  
  /**
   * L'armure équipée
   * @type {Item5e|null}
   */
  get armor() {
    return this.system.attributes?.ac?.equippedArmor ?? null;
  }
  
  /**
   * Le bouclier équipé
   * @type {Item5e|null}
   */
  get shield() {
    return this.system.attributes?.ac?.equippedShield ?? null;
  }
  
  /* -------------------------------------------- */
  /*  Préparation des Données                     */
  /* -------------------------------------------- */
  
  /** @override */
  prepareData() {
    // Nettoyer les caches
    this._preparationWarnings = [];
    this.labels = {};
    
    // Appeler la méthode parente
    super.prepareData();
    
    // Préparer les données finales des items
    this.items.forEach(item => item.prepareFinalAttributes());
  }
  
  /** @override */
  prepareBaseData() {
    // Préparation avant les effets actifs
    // Calculer le niveau, la maîtrise de base, etc.
  }
  
  /** @override */
  prepareEmbeddedDocuments() {
    // Préparer les items et effets imbriqués
    super.prepareEmbeddedDocuments();
  }
  
  /** @override */
  applyActiveEffects() {
    // Permettre au DataModel de préparer les données embedded
    if ( this.system?.prepareEmbeddedData ) {
      this.system.prepareEmbeddedData();
    }
    return super.applyActiveEffects();
  }
  
  /** @override */
  prepareDerivedData() {
    // Calculer les données dérivées après effets
    // Modificateurs finaux, bonus, etc.
  }
  
  /* -------------------------------------------- */
  /*  Méthodes de Jet de Dés                      */
  /* -------------------------------------------- */
  
  /**
   * Effectuer un jet de compétence
   * @param {object} config - Configuration du jet
   * @returns {Promise<D20Roll[]|null>}
   */
  async rollSkill(config = {}, dialog = {}, message = {}) {
    if ( !this.system.skills ) return null;
    
    const skill = this.system.skills[config.skill];
    const ability = this.system.abilities[skill.ability];
    
    // Construire le jet
    const roll = await CONFIG.Dice.D20Roll.build({
      parts: ["1d20", "@mod", "@prof"],
      data: {
        mod: ability.mod,
        prof: skill.prof.term
      },
      ...config
    });
    
    return roll;
  }
  
  /**
   * Effectuer un jet de sauvegarde
   */
  async rollSavingThrow(config = {}, dialog = {}, message = {}) {
    const ability = this.system.abilities?.[config.ability];
    if ( !ability ) return null;
    
    const rollConfig = {
      parts: ["1d20", "@mod"],
      data: { mod: ability.save },
      ...config
    };
    
    return CONFIG.Dice.D20Roll.build(rollConfig, dialog, message);
  }
  
  /* -------------------------------------------- */
  /*  Gestion des Points de Vie                   */
  /* -------------------------------------------- */
  
  /**
   * Appliquer des dégâts à l'acteur
   * @param {number} amount - Montant de dégâts
   * @returns {Promise<Actor5e>}
   */
  async applyDamage(amount, options = {}) {
    const hp = this.system.attributes.hp;
    if ( !hp ) return this;
    
    // Calculer les dégâts réels (résistances, etc.)
    const { value, temp } = this.calculateDamage(amount, options);
    
    // Mettre à jour
    const updates = {
      "system.attributes.hp.temp": Math.max(0, hp.temp - temp),
      "system.attributes.hp.value": Math.max(0, hp.value - value)
    };
    
    await this.update(updates);
    return this;
  }
  
  /* -------------------------------------------- */
  /*  Hooks de Cycle de Vie                       */
  /* -------------------------------------------- */
  
  /** @override */
  async _preCreate(data, options, user) {
    if ( await super._preCreate(data, options, user) === false ) return false;
    
    // Configurer le token prototype par défaut
    const prototypeToken = {};
    
    if ( this.type === "character" ) {
      Object.assign(prototypeToken, {
        sight: { enabled: true },
        actorLink: true,
        disposition: CONST.TOKEN_DISPOSITIONS.FRIENDLY
      });
    }
    
    // Appliquer la taille par défaut
    const size = CONFIG.DND5E.actorSizes[this.system.traits?.size]?.token ?? 1;
    prototypeToken.width = size;
    prototypeToken.height = size;
    
    this.updateSource({ prototypeToken });
  }
  
  /** @override */
  async _preUpdate(changed, options, user) {
    if ( await super._preUpdate(changed, options, user) === false ) return false;
    
    // Ajuster la taille du token si la taille change
    const newSize = foundry.utils.getProperty(changed, "system.traits.size");
    if ( newSize && newSize !== this.system.traits?.size ) {
      const size = CONFIG.DND5E.actorSizes[newSize].token ?? 1;
      changed.prototypeToken ??= {};
      changed.prototypeToken.width = size;
      changed.prototypeToken.height = size;
    }
  }
}
```

## Extension de Item

### Structure de Base

```javascript
// module/documents/item.mjs
import SystemDocumentMixin from "./mixins/document.mjs";

export default class Item5e extends SystemDocumentMixin(Item) {
  
  static DEFAULT_ICON = "systems/dnd5e/icons/svg/documents/item.svg";
  
  /* -------------------------------------------- */
  /*  Propriétés                                  */
  /* -------------------------------------------- */
  
  /**
   * L'item est-il équipable ?
   * @type {boolean}
   */
  get isEquippable() {
    return this.system.equipped !== undefined;
  }
  
  /**
   * L'item a-t-il des utilisations limitées ?
   * @type {boolean}
   */
  get hasLimitedUses() {
    return this.system.uses?.max > 0;
  }
  
  /**
   * L'item nécessite-t-il un jet d'attaque ?
   * @type {boolean}
   */
  get hasAttack() {
    return this.system.hasAttack ?? false;
  }
  
  /**
   * Conteneur parent si l'item est dans un sac
   * @type {Item5e|void}
   */
  get container() {
    if ( !this.system.container ) return;
    if ( this.isEmbedded ) return this.actor.items.get(this.system.container);
    return game.items.get(this.system.container);
  }
  
  /**
   * La classe associée (pour les sous-classes)
   * @type {Item5e|null}
   */
  get class() {
    if ( this.type !== "subclass" || !this.isEmbedded ) return null;
    return this.parent.items.find(i => 
      i.type === "class" && i.identifier === this.system.classIdentifier
    );
  }
  
  /* -------------------------------------------- */
  /*  Préparation des Données                     */
  /* -------------------------------------------- */
  
  /** @override */
  prepareData() {
    super.prepareData();
    this.labels = {};
  }
  
  /** @override */
  prepareDerivedData() {
    super.prepareDerivedData();
    
    // Calculer les propriétés dérivées
    this._prepareLabels();
  }
  
  /**
   * Préparer les labels d'affichage
   * @private
   */
  _prepareLabels() {
    const labels = this.labels;
    
    // Type d'activation
    const activation = this.system.activation;
    if ( activation?.type ) {
      labels.activation = [
        activation.cost,
        CONFIG.DND5E.abilityActivationTypes[activation.type]
      ].filterJoin(" ");
    }
    
    // Portée
    const range = this.system.range;
    if ( range?.value ) {
      labels.range = range.units === "touch" 
        ? CONFIG.DND5E.rangeTypes.touch 
        : `${range.value} ${range.units}`;
    }
  }
  
  /* -------------------------------------------- */
  /*  Utilisation de l'Item                       */
  /* -------------------------------------------- */
  
  /**
   * Utiliser l'item
   * @param {object} config - Options de configuration
   * @returns {Promise<ChatMessage|object|void>}
   */
  async use(config = {}, dialog = {}, message = {}) {
    // Vérifier les utilisations restantes
    if ( this.hasLimitedUses ) {
      const uses = this.system.uses;
      if ( uses.value <= 0 ) {
        ui.notifications.warn(game.i18n.format("DND5E.ItemNoUses", {
          name: this.name
        }));
        return;
      }
    }
    
    // Consommer une utilisation
    if ( config.consumeUsage !== false && this.hasLimitedUses ) {
      await this.update({
        "system.uses.value": this.system.uses.value - 1
      });
    }
    
    // Créer le message de chat
    const chatData = {
      speaker: ChatMessage.getSpeaker({ actor: this.actor }),
      content: await this._getChatContent(),
      "flags.dnd5e.item": { id: this.id }
    };
    
    return ChatMessage.create(chatData);
  }
  
  /**
   * Effectuer un jet d'attaque
   * @returns {Promise<D20Roll>}
   */
  async rollAttack(options = {}) {
    if ( !this.hasAttack ) {
      throw new Error(`${this.name} does not have an attack roll.`);
    }
    
    const rollData = this.getRollData();
    
    const roll = await CONFIG.Dice.D20Roll.build({
      parts: ["1d20", "@mod", "@prof", "@bonus"],
      data: rollData,
      title: game.i18n.format("DND5E.AttackRollPrompt", { item: this.name })
    });
    
    return roll;
  }
  
  /**
   * Effectuer un jet de dégâts
   * @returns {Promise<DamageRoll>}
   */
  async rollDamage(options = {}) {
    const parts = this.system.damage?.parts ?? [];
    if ( !parts.length ) {
      throw new Error(`${this.name} does not have damage configured.`);
    }
    
    const rollData = this.getRollData();
    const formula = parts.map(p => p[0]).join(" + ");
    
    return new CONFIG.Dice.DamageRoll(formula, rollData).evaluate();
  }
  
  /* -------------------------------------------- */
  /*  Hooks de Cycle de Vie                       */
  /* -------------------------------------------- */
  
  /** @override */
  async _preCreate(data, options, user) {
    if ( await super._preCreate(data, options, user) === false ) return false;
    
    // Définir une icône par défaut selon le type
    if ( !data.img || data.img === this.constructor.DEFAULT_ICON ) {
      const defaultIcon = CONFIG.DND5E.defaultItemIcons?.[this.type];
      if ( defaultIcon ) {
        this.updateSource({ img: defaultIcon });
      }
    }
  }
  
  /** @override */
  _onCreate(data, options, userId) {
    super._onCreate(data, options, userId);
    
    // Actions après création
    if ( this.isEmbedded && this.type === "class" ) {
      // Définir comme classe principale si c'est la première
      if ( !this.parent.system.details?.originalClass ) {
        this.parent.update({ 
          "system.details.originalClass": this.id 
        });
      }
    }
  }
  
  /** @override */
  async _preDelete(options, user) {
    if ( await super._preDelete(options, user) === false ) return false;
    
    // Empêcher la suppression d'items verrouillés
    if ( this.flags.dnd5e?.locked ) {
      ui.notifications.warn("This item is locked and cannot be deleted.");
      return false;
    }
  }
}
```

## Extension de ActiveEffect

```javascript
// module/documents/active-effect.mjs
export default class ActiveEffect5e extends ActiveEffect {
  
  /**
   * L'effet est-il temporaire ?
   * @type {boolean}
   */
  get isTemporary() {
    const dominated = CONFIG.DND5E.actorConditions.every(c => 
      !this.statuses.has(c)
    );
    return super.isTemporary && dominated;
  }
  
  /**
   * L'effet supprime-t-il les statuts ?
   * @type {boolean}
   */
  get isSuppressed() {
    if ( !this.parent?.isEmbedded ) return false;
    const item = this.parent;
    const actor = item?.parent;
    
    // Supprimé si l'item n'est pas équipé
    if ( item.system.equipped === false ) return true;
    
    // Supprimé si concentration perdue
    if ( this.flags.dnd5e?.concentration && !actor?.concentration.effects.has(this) ) {
      return true;
    }
    
    return false;
  }
  
  /* -------------------------------------------- */
  /*  Méthodes Statiques                          */
  /* -------------------------------------------- */
  
  /**
   * Créer un effet de concentration
   * @param {Item5e} item - L'item source
   * @returns {object} - Données de l'effet
   */
  static createConcentrationEffectData(item, data = {}) {
    return foundry.utils.mergeObject({
      name: game.i18n.format("DND5E.ConcentrationOn", { name: item.name }),
      img: item.img,
      origin: item.uuid,
      duration: this._getConcentrationDuration(item),
      statuses: [CONFIG.specialStatusEffects.CONCENTRATING],
      "flags.dnd5e": {
        type: "concentration",
        item: { id: item.id, data: item.toObject() }
      }
    }, data);
  }
  
  /* -------------------------------------------- */
  /*  Hooks de Cycle de Vie                       */
  /* -------------------------------------------- */
  
  /** @override */
  _preCreate(data, options, user) {
    if ( super._preCreate(data, options, user) === false ) return false;
    
    // Vérifier la limite de concentration
    if ( this.statuses.has(CONFIG.specialStatusEffects.CONCENTRATING) ) {
      const actor = this.parent;
      const limit = actor?.system.attributes?.concentration?.limit ?? 1;
      
      if ( actor?.concentration.effects.size >= limit ) {
        // Demander quelle concentration terminer
        return this._promptConcentrationChoice(actor);
      }
    }
  }
}
```

## Extension de ChatMessage

```javascript
// module/documents/chat-message.mjs
export default class ChatMessage5e extends ChatMessage {
  
  /**
   * Le message contient-il un jet de dégâts ?
   * @type {boolean}
   */
  get hasDamage() {
    return this.rolls.some(r => r instanceof CONFIG.Dice.DamageRoll);
  }
  
  /**
   * Activer les écouteurs sur les messages de chat
   */
  static activateListeners() {
    document.body.addEventListener("click", this._onClickButton.bind(this));
  }
  
  /**
   * Gérer les clics sur les boutons
   * @param {PointerEvent} event
   */
  static async _onClickButton(event) {
    const button = event.target.closest("[data-action]");
    if ( !button ) return;
    
    const action = button.dataset.action;
    const messageId = button.closest("[data-message-id]")?.dataset.messageId;
    const message = game.messages.get(messageId);
    
    switch ( action ) {
      case "apply-damage":
        return this._onApplyDamage(message, button);
      case "reroll":
        return this._onReroll(message, button);
    }
  }
  
  /**
   * Appliquer les dégâts aux cibles
   */
  static async _onApplyDamage(message, button) {
    const damage = parseInt(button.dataset.damage);
    const targets = game.user.targets;
    
    for ( const token of targets ) {
      await token.actor?.applyDamage(damage);
    }
  }
}
```

## Extension de Combat

```javascript
// module/documents/combat.mjs
export default class Combat5e extends Combat {
  
  /** @override */
  async rollInitiative(ids, options = {}) {
    // Logique personnalisée d'initiative
    const formula = options.formula ?? this._getInitiativeFormula();
    
    for ( const id of ids ) {
      const combatant = this.combatants.get(id);
      const roll = await combatant.actor?.getInitiativeRoll(options);
      
      if ( roll ) {
        await roll.evaluate();
        await combatant.update({ initiative: roll.total });
      }
    }
    
    return this;
  }
  
  /** @override */
  async nextTurn() {
    // Actions de fin de tour
    const current = this.combatant;
    if ( current ) {
      await this._handleTurnEnd(current);
    }
    
    // Passer au tour suivant
    const result = await super.nextTurn();
    
    // Actions de début de tour
    const next = this.combatant;
    if ( next ) {
      await this._handleTurnStart(next);
    }
    
    return result;
  }
  
  /**
   * Gérer la fin d'un tour
   */
  async _handleTurnEnd(combatant) {
    const actor = combatant.actor;
    if ( !actor ) return;
    
    // Régénération, effets de fin de tour, etc.
    const effects = actor.effects.filter(e => 
      e.flags.dnd5e?.duration?.endOfTurn
    );
    
    for ( const effect of effects ) {
      await effect.delete();
    }
  }
}
```

## Mixin de Document Système

Un pattern utile pour partager du code entre Actor et Item :

```javascript
// module/documents/mixins/document.mjs
export default function SystemDocumentMixin(Base) {
  return class SystemDocument extends Base {
    
    /**
     * Données de roll pour les formules
     * @returns {object}
     */
    getRollData() {
      const data = { ...super.getRollData() };
      
      // Ajouter les données système
      if ( this.system?.getRollData ) {
        Object.assign(data, this.system.getRollData());
      }
      
      return data;
    }
    
    /**
     * Vérifier si une propriété est dérivée des effets
     * @param {string} key - Chemin de la propriété
     * @returns {boolean}
     */
    isPropertyDerived(key) {
      return this.overrides?.[key] !== undefined;
    }
    
    /**
     * Obtenir l'attribution d'une valeur
     * @param {string} key - Chemin de la propriété
     * @returns {object[]} - Sources de la valeur
     */
    getPropertyAttribution(key) {
      // Retourner les effets qui modifient cette propriété
      return this.effects.filter(e => 
        e.changes.some(c => c.key === key)
      );
    }
  };
}
```

## Enregistrement des Extensions

```javascript
// main.mjs
Hooks.once("init", function() {
  // Enregistrer les classes de documents
  CONFIG.Actor.documentClass = Actor5e;
  CONFIG.Item.documentClass = Item5e;
  CONFIG.ActiveEffect.documentClass = ActiveEffect5e;
  CONFIG.ChatMessage.documentClass = ChatMessage5e;
  CONFIG.Combat.documentClass = Combat5e;
  CONFIG.Combatant.documentClass = Combatant5e;
  
  // Optionnel : Enregistrer des sous-classes de tokens
  CONFIG.Token.documentClass = TokenDocument5e;
  CONFIG.Token.objectClass = Token5e;
});
```

## Bonnes Pratiques

1. **Utiliser `super`** : Toujours appeler les méthodes parentes
2. **Retour `false`** : Dans `_preCreate`, `_preUpdate`, `_preDelete` pour annuler
3. **Validation côté client** : Vérifier les données dans `_pre*` hooks
4. **Effets secondaires** : Réserver les effets secondaires pour `_on*` hooks
5. **Cache** : Utiliser `_lazy` ou des getters mémoïsés pour les calculs coûteux
6. **Mixins** : Factoriser le code commun dans des mixins

## Liens vers la Documentation Officielle

- [Document Class](https://foundryvtt.com/api/classes/foundry.abstract.Document.html)
- [Actor](https://foundryvtt.com/api/classes/Actor.html)
- [Item](https://foundryvtt.com/api/classes/Item.html)
- [ActiveEffect](https://foundryvtt.com/api/classes/ActiveEffect.html)

---

*Documentation suivante : [04-applications.md](./04-applications.md) - Feuilles et Applications (AppV2)*
