# DataModels et TypeDataModel

> Documentation technique pour FoundryVTT v13

## Vue d'ensemble

Les **DataModels** sont le système moderne de définition et validation des données dans FoundryVTT. Ils remplacent l'ancien fichier `template.json` et offrent :

- **Typage fort** des propriétés
- **Validation automatique** des données
- **Valeurs par défaut** intelligentes
- **Migrations** intégrées
- **Autocomplétion** dans les IDE

## Hiérarchie des Classes

```
foundry.abstract.DataModel
    │
    ├── foundry.abstract.TypeDataModel    ← Pour les documents avec sous-types
    │       │
    │       └── SystemDataModel (personnalisé)
    │               │
    │               ├── CharacterData
    │               ├── NPCData
    │               └── WeaponData
    │
    └── Autres DataModels (settings, embedded, etc.)
```

## DataModel vs TypeDataModel

| Aspect | DataModel | TypeDataModel |
|--------|-----------|---------------|
| **Usage** | Données génériques | Documents système (Actor, Item) |
| **Accès parent** | `this.parent` | `this.parent` (le Document) |
| **Sous-types** | Non | Oui (character, npc, weapon...) |
| **Propriété `type`** | Non gérée | Gérée automatiquement |

### Quand utiliser lequel ?

- **TypeDataModel** : Pour les données `system` des Actor, Item, ActiveEffect, ChatMessage, JournalEntryPage
- **DataModel** : Pour les données imbriquées, les settings, les structures personnalisées

## Définir un Schéma avec `defineSchema()`

La méthode statique `defineSchema()` retourne un objet décrivant la structure des données.

### Syntaxe de Base

```javascript
import { foundry } from "foundry";

const { 
  StringField, NumberField, BooleanField, 
  SchemaField, ArrayField, SetField,
  HTMLField, FilePathField
} = foundry.data.fields;

export default class CharacterData extends foundry.abstract.TypeDataModel {
  
  static defineSchema() {
    return {
      // Champ texte simple
      name: new StringField({
        required: true,
        blank: false,
        initial: "Nouveau Personnage"
      }),
      
      // Champ numérique
      level: new NumberField({
        required: true,
        nullable: false,
        integer: true,
        min: 1,
        max: 20,
        initial: 1
      }),
      
      // Champ booléen
      isNPC: new BooleanField({
        initial: false
      }),
      
      // Champ HTML enrichi
      biography: new HTMLField({
        required: true,
        blank: true
      }),
      
      // Objet imbriqué
      attributes: new SchemaField({
        hp: new SchemaField({
          value: new NumberField({ integer: true, min: 0, initial: 10 }),
          max: new NumberField({ integer: true, min: 0, initial: 10 }),
          temp: new NumberField({ integer: true, min: 0, initial: 0 })
        }),
        ac: new NumberField({ integer: true, min: 0, initial: 10 })
      }),
      
      // Tableau
      languages: new ArrayField(
        new StringField({ blank: false })
      ),
      
      // Ensemble (valeurs uniques)
      proficiencies: new SetField(
        new StringField({ blank: false })
      )
    };
  }
}
```

## Types de Champs Disponibles

### Champs Primitifs

| Champ | Description | Options Clés |
|-------|-------------|--------------|
| `StringField` | Texte | `blank`, `trim`, `choices` |
| `NumberField` | Nombre | `integer`, `positive`, `min`, `max`, `step` |
| `BooleanField` | Booléen | - |
| `HTMLField` | HTML enrichi | `blank` |
| `FilePathField` | Chemin de fichier | `categories` |
| `JSONField` | Données JSON | - |
| `ColorField` | Couleur hexadécimale | - |
| `IntegerSortField` | Tri entier | - |

### Champs Composés

| Champ | Description | Usage |
|-------|-------------|-------|
| `SchemaField` | Objet avec sous-schéma | Grouper des champs liés |
| `ArrayField` | Tableau d'éléments | Listes dynamiques |
| `SetField` | Ensemble (valeurs uniques) | Tags, proficiencies |
| `EmbeddedDataField` | DataModel imbriqué | Structures réutilisables |
| `MappingField` | Dictionnaire clé/valeur | Objets dynamiques |

### Champs Spéciaux

| Champ | Description | Usage |
|-------|-------------|-------|
| `DocumentIdField` | ID de document | Références internes |
| `DocumentUUIDField` | UUID de document | Références cross-compendium |
| `ForeignDocumentField` | Référence à un autre document | Relations |

## Options Communes des Champs

```javascript
new NumberField({
  // ═══════════════════════════════════════
  // OPTIONS GÉNÉRALES (tous les champs)
  // ═══════════════════════════════════════
  required: true,          // Le champ doit exister
  nullable: false,         // Peut être null ?
  initial: 0,              // Valeur par défaut
  readonly: false,         // Modification interdite après création
  label: "DND5E.Level",    // Clé de traduction
  hint: "DND5E.LevelHint", // Aide contextuelle
  
  // Validation personnalisée
  validate: (value) => value > 0,
  validationError: "La valeur doit être positive",
  
  // ═══════════════════════════════════════
  // OPTIONS SPÉCIFIQUES (NumberField)
  // ═══════════════════════════════════════
  integer: true,           // Nombre entier uniquement
  positive: true,          // Strictement positif
  min: 0,                  // Valeur minimale
  max: 100,                // Valeur maximale
  step: 1,                 // Incrément
  
  // ═══════════════════════════════════════
  // OPTIONS SPÉCIFIQUES (StringField)
  // ═══════════════════════════════════════
  blank: true,             // Chaîne vide autorisée
  trim: true,              // Supprimer espaces
  choices: ["a", "b", "c"] // Valeurs autorisées
});
```

## Exemple Complet : Character Data (dnd5e)

Extrait simplifié de `module/data/actor/character.mjs` :

```javascript
import CreatureTemplate from "./templates/creature.mjs";
import AttributesFields from "./templates/attributes.mjs";
import DetailsFields from "./templates/details.mjs";

const { 
  ArrayField, BooleanField, HTMLField, 
  NumberField, SchemaField, SetField, StringField 
} = foundry.data.fields;

export default class CharacterData extends CreatureTemplate {

  // Type système pour l'enregistrement
  static _systemType = "character";

  // Métadonnées du modèle
  static metadata = Object.freeze({
    supportsAdvancement: true
  });

  // Définition du schéma
  static defineSchema() {
    return this.mergeSchema(super.defineSchema(), {
      
      // Attributs du personnage
      attributes: new SchemaField({
        ...AttributesFields.common,
        ...AttributesFields.creature,
        
        hp: new SchemaField({
          value: new NumberField({ integer: true, min: 0, initial: 0 }),
          max: new NumberField({ nullable: true, integer: true, min: 0 }),
          temp: new NumberField({ integer: true, min: 0, initial: 0 }),
          bonuses: new SchemaField({
            level: new FormulaField({ deterministic: true }),
            overall: new FormulaField({ deterministic: true })
          })
        }),
        
        death: new SchemaField({
          success: new NumberField({ integer: true, min: 0, initial: 0 }),
          failure: new NumberField({ integer: true, min: 0, initial: 0 })
        }),
        
        inspiration: new BooleanField({ initial: false })
      }),
      
      // Détails du personnage
      details: new SchemaField({
        ...DetailsFields.common,
        background: new StringField({ required: true }),
        xp: new SchemaField({
          value: new NumberField({ integer: true, min: 0, initial: 0 })
        }),
        appearance: new StringField(),
        trait: new StringField(),
        gender: new StringField(),
        age: new StringField()
      }),
      
      // Ressources personnalisables
      resources: new SchemaField({
        primary: makeResourceField(),
        secondary: makeResourceField(),
        tertiary: makeResourceField()
      }),
      
      // Favoris
      favorites: new ArrayField(new SchemaField({
        type: new StringField({ required: true, blank: false }),
        id: new StringField({ required: true, blank: false }),
        sort: new IntegerSortField()
      }))
    });
  }
}

// Fonction helper pour créer des champs de ressource
function makeResourceField() {
  return new SchemaField({
    value: new NumberField({ integer: true, initial: 0 }),
    max: new NumberField({ integer: true, initial: 0 }),
    sr: new BooleanField({ initial: false }),  // Récupération repos court
    lr: new BooleanField({ initial: false }),  // Récupération repos long
    label: new StringField()
  });
}
```

## Système de Templates (Mixins)

dnd5e utilise un système de templates pour réutiliser des schémas entre différents types.

### Créer un Template

```javascript
// module/data/actor/templates/common.mjs
export default class CommonTemplate extends foundry.abstract.TypeDataModel {
  
  static defineSchema() {
    return {
      abilities: new MappingField(new SchemaField({
        value: new NumberField({ integer: true, min: 0, max: 30, initial: 10 }),
        proficient: new NumberField({ min: 0, max: 2, step: 0.5, initial: 0 }),
        bonuses: new SchemaField({
          check: new FormulaField(),
          save: new FormulaField()
        })
      }), {
        initialKeys: CONFIG.DND5E.abilities,
        initialValue: this._initialAbilityValue
      }),
      
      currency: new SchemaField({
        pp: new NumberField({ integer: true, min: 0, initial: 0 }),
        gp: new NumberField({ integer: true, min: 0, initial: 0 }),
        ep: new NumberField({ integer: true, min: 0, initial: 0 }),
        sp: new NumberField({ integer: true, min: 0, initial: 0 }),
        cp: new NumberField({ integer: true, min: 0, initial: 0 })
      })
    };
  }
}
```

### Utiliser la Méthode `mixin()`

```javascript
// module/data/actor/character.mjs
import CommonTemplate from "./templates/common.mjs";
import CreatureTemplate from "./templates/creature.mjs";
import AttributesTemplate from "./templates/attributes.mjs";

// Mixer plusieurs templates
export default class CharacterData extends SystemDataModel.mixin(
  CommonTemplate,
  CreatureTemplate, 
  AttributesTemplate
) {
  static defineSchema() {
    // Les schémas des templates sont automatiquement fusionnés
    return this.mergeSchema(super.defineSchema(), {
      // Champs spécifiques au personnage
      level: new NumberField({ integer: true, min: 1, initial: 1 })
    });
  }
}
```

## Champs Personnalisés

Vous pouvez créer vos propres types de champs.

### Exemple : FormulaField (dnd5e)

```javascript
// module/data/fields/formula-field.mjs
export default class FormulaField extends foundry.data.fields.StringField {
  
  // Options spécifiques
  static get _defaults() {
    return foundry.utils.mergeObject(super._defaults, {
      deterministic: false  // Formule sans dés ?
    });
  }
  
  // Validation personnalisée
  _validateType(value) {
    if ( value === "" ) return;
    
    // Vérifier que c'est une formule de dés valide
    try {
      const roll = new Roll(value);
      if ( this.options.deterministic && !roll.isDeterministic ) {
        throw new Error("Formula must be deterministic");
      }
    } catch(err) {
      throw new foundry.data.validation.DataModelValidationError(err.message);
    }
  }
}
```

### Exemple : MappingField (dnd5e)

```javascript
// module/data/fields/mapping-field.mjs
export default class MappingField extends foundry.data.fields.ObjectField {
  
  constructor(model, options={}) {
    super(options);
    this.model = model;           // Modèle pour chaque valeur
    this.initialKeys = options.initialKeys;    // Clés initiales
    this.initialValue = options.initialValue;  // Fonction de valeur initiale
  }
  
  initialize(value, model, options={}) {
    const result = {};
    for ( const [key, data] of Object.entries(value ?? {}) ) {
      result[key] = this.model.initialize(data, model, options);
    }
    return result;
  }
  
  // Créer les clés initiales
  getInitialValue(data) {
    const initial = {};
    if ( !this.initialKeys ) return initial;
    
    const keys = typeof this.initialKeys === "function" 
      ? this.initialKeys() 
      : this.initialKeys;
      
    for ( const key of Object.keys(keys) ) {
      const value = this.model.getInitialValue();
      initial[key] = this.initialValue?.(key, value) ?? value;
    }
    return initial;
  }
}
```

## Validation des Données

### Validation Automatique

FoundryVTT valide automatiquement les données selon le schéma :

```javascript
// Ces données seront validées
const actor = await Actor.create({
  name: "Test",
  type: "character",
  system: {
    level: 25  // ❌ Erreur : max est 20
  }
});
```

### Validation Personnalisée avec `validate()`

```javascript
static defineSchema() {
  return {
    hp: new SchemaField({
      value: new NumberField({ 
        validate: (value, options) => {
          const max = options?.source?.max ?? Infinity;
          if ( value > max ) {
            throw new Error("HP cannot exceed max HP");
          }
        }
      }),
      max: new NumberField({ min: 0 })
    })
  };
}
```

### Validation Jointe avec `validateJoint()`

```javascript
// Valider plusieurs champs ensemble
static validateJoint(data) {
  super.validateJoint(data);
  
  if ( data.hp.value > data.hp.max ) {
    throw new foundry.data.validation.DataModelValidationError(
      "Current HP cannot exceed maximum HP"
    );
  }
}
```

## Migration des Données

### Méthode `migrateData()`

Transforme les anciennes données vers le nouveau format :

```javascript
static migrateData(source) {
  // Migrer l'ancien format de vitesse
  if ( "speed" in source && typeof source.speed === "number" ) {
    source.attributes ??= {};
    source.attributes.movement ??= {};
    source.attributes.movement.walk = source.speed;
    delete source.speed;
  }
  
  // Migrer les anciennes compétences
  if ( source.skills?.acr?.mod !== undefined ) {
    for ( const [key, skill] of Object.entries(source.skills) ) {
      skill.value = skill.mod >= 0 ? Math.floor(skill.mod / 2) : 0;
      delete skill.mod;
    }
  }
  
  return super.migrateData(source);
}
```

### Méthode `shimData()` (Rétrocompatibilité)

Ajoute des propriétés virtuelles pour la compatibilité :

```javascript
static shimData(data, options) {
  // Ajouter un getter pour l'ancien chemin
  this._addDataFieldShim(data, "speed", "attributes.movement.walk", {
    since: "3.0",
    until: "4.0"
  });
  
  return super.shimData(data, options);
}
```

## Préparation des Données

### Méthodes de Préparation

Les DataModels peuvent implémenter des méthodes de préparation :

```javascript
export default class CharacterData extends CreatureTemplate {
  
  // Préparation de base (avant effets actifs)
  prepareBaseData() {
    // Calculer le niveau total
    this.details.level = 0;
    for ( const item of this.parent.items ) {
      if ( item.type === "class" ) {
        this.details.level += item.system.levels;
      }
    }
    
    // Calculer le bonus de maîtrise
    this.attributes.prof = Math.floor((this.details.level + 7) / 4);
  }
  
  // Préparation après effets actifs
  prepareDerivedData() {
    const rollData = this.parent.getRollData({ deterministic: true });
    
    // Calculer les modificateurs d'abilities
    for ( const [id, ability] of Object.entries(this.abilities) ) {
      ability.mod = Math.floor((ability.value - 10) / 2);
    }
    
    // Calculer l'AC
    this.attributes.ac.value = 10 + (this.abilities.dex?.mod ?? 0);
  }
  
  // Données de roll
  getRollData() {
    return {
      level: this.details.level,
      prof: this.attributes.prof,
      abilities: Object.fromEntries(
        Object.entries(this.abilities).map(([k, v]) => [k, v.mod])
      )
    };
  }
}
```

## Enregistrement des DataModels

Dans le hook `init`, enregistrez vos DataModels :

```javascript
// module/data/actor/_module.mjs
import CharacterData from "./character.mjs";
import NPCData from "./npc.mjs";
import VehicleData from "./vehicle.mjs";

export const config = {
  character: CharacterData,
  npc: NPCData,
  vehicle: VehicleData
};

// main.mjs
Hooks.once("init", function() {
  CONFIG.Actor.dataModels = dataModels.actor.config;
  CONFIG.Item.dataModels = dataModels.item.config;
});
```

## Bonnes Pratiques

1. **Typage strict** : Utilisez `nullable: false` quand possible
2. **Valeurs initiales** : Toujours définir `initial`
3. **Validation** : Utiliser `min`, `max`, `choices` pour contraindre
4. **Labels** : Ajouter `label` pour l'internationalisation
5. **Templates** : Factoriser avec des mixins
6. **Migrations** : Prévoir l'évolution du schéma

## Liens vers la Documentation Officielle

- [DataModel API](https://foundryvtt.com/api/classes/foundry.abstract.DataModel.html)
- [TypeDataModel API](https://foundryvtt.com/api/classes/foundry.abstract.TypeDataModel.html)
- [Data Fields](https://foundryvtt.com/api/modules/foundry.data.fields.html)

---

*Documentation suivante : [03-documents.md](./03-documents.md) - Documents et Extensions*
