# Architecture d'un Système FoundryVTT v13

> Documentation technique pour développeurs créant un système FoundryVTT de zéro.

## Vue d'ensemble

Un **système** FoundryVTT est un module spécialisé qui définit les règles de jeu, les structures de données des personnages et objets, ainsi que les interfaces utilisateur associées. Contrairement aux modules qui étendent les fonctionnalités, un système constitue le socle fondamental d'un monde de jeu.

## Arborescence Type d'un Système

```
mon-systeme/
├── system.json              # Manifeste du système (obligatoire)
├── mon-systeme.mjs          # Point d'entrée JavaScript principal
├── mon-systeme.css          # Feuilles de style compilées
├── module/                  # Code source JavaScript/TypeScript
│   ├── _module.mjs          # Export centralisé des sous-modules
│   ├── config.mjs           # Configuration globale du système
│   ├── settings.mjs         # Paramètres système
│   ├── utils.mjs            # Fonctions utilitaires
│   ├── data/                # DataModels (schémas de données)
│   │   ├── _module.mjs
│   │   ├── actor/           # Modèles pour les Actors
│   │   │   ├── character.mjs
│   │   │   ├── npc.mjs
│   │   │   └── templates/   # Templates réutilisables
│   │   ├── item/            # Modèles pour les Items
│   │   │   ├── weapon.mjs
│   │   │   ├── spell.mjs
│   │   │   └── templates/
│   │   └── shared/          # Champs partagés
│   ├── documents/           # Extensions de Documents
│   │   ├── actor.mjs
│   │   ├── item.mjs
│   │   ├── active-effect.mjs
│   │   └── chat-message.mjs
│   ├── applications/        # Feuilles et interfaces (AppV2)
│   │   ├── actor/
│   │   ├── item/
│   │   └── sheets/
│   ├── dice/                # Logique de jets de dés personnalisés
│   └── canvas/              # Extensions du canvas
├── templates/               # Templates Handlebars (.hbs)
│   ├── actors/
│   ├── items/
│   └── chat/
├── packs/                   # Compendiums (données précompilées)
│   ├── monsters/
│   └── spells/
├── lang/                    # Fichiers de traduction
│   └── en.json
├── icons/                   # Icônes du système
├── fonts/                   # Polices personnalisées
└── less/ ou scss/           # Sources des styles
```

## Rôle des Fichiers et Dossiers Clés

### Fichiers Racine

| Fichier | Rôle |
|---------|------|
| `system.json` | Manifeste définissant les métadonnées, dépendances, types de documents et compendiums |
| `*.mjs` (point d'entrée) | Initialise le système, enregistre les hooks, configure les classes de documents |
| `*.css` | Styles compilés pour l'interface utilisateur |

### Dossier `module/`

| Sous-dossier | Rôle |
|--------------|------|
| `config.mjs` | Objet de configuration globale (ex: `CONFIG.DND5E`) contenant les listes, constantes, et options du système |
| `data/` | **DataModels** - Schémas de validation et structure des données (TypeDataModel) |
| `documents/` | Extensions des classes Document de Foundry (Actor, Item, ActiveEffect, etc.) |
| `applications/` | Interfaces utilisateur - feuilles de personnage, fiches d'objets (Application V2) |
| `dice/` | Classes de jets de dés personnalisés (ex: D20Roll, DamageRoll) |
| `canvas/` | Extensions du canvas (tokens, rulers, templates) |

### Dossier `templates/`

Templates Handlebars (`.hbs`) pour le rendu HTML des feuilles et messages de chat.

### Dossier `packs/`

Compendiums contenant les données de référence (monstres, sorts, équipements). Peuvent être au format LevelDB ou JSON source.

## Cycle de Vie d'un Système

FoundryVTT exécute les systèmes selon un cycle de vie précis, orchestré par des **Hooks** :

```
┌─────────────────────────────────────────────────────────────────┐
│                        CHARGEMENT                               │
├─────────────────────────────────────────────────────────────────┤
│  1. Le navigateur charge le manifeste system.json               │
│  2. Les fichiers esmodules et styles sont chargés               │
└─────────────────────────────────────────────────────────────────┘
                              │
                              ▼
┌─────────────────────────────────────────────────────────────────┐
│                     HOOK: "init"                                │
├─────────────────────────────────────────────────────────────────┤
│  - Enregistrer les classes de documents dans CONFIG             │
│  - Enregistrer les DataModels                                   │
│  - Configurer les classes de rolls                              │
│  - Enregistrer les paramètres système (game.settings)           │
│  - Enregistrer les feuilles (DocumentSheetConfig)               │
│  - Précharger les templates Handlebars                          │
│  - Les données du monde ne sont PAS encore disponibles          │
└─────────────────────────────────────────────────────────────────┘
                              │
                              ▼
┌─────────────────────────────────────────────────────────────────┐
│                     HOOK: "i18nInit"                            │
├─────────────────────────────────────────────────────────────────┤
│  - Le système d'internationalisation est prêt                   │
│  - Pré-localiser les configurations (labels, options)           │
└─────────────────────────────────────────────────────────────────┘
                              │
                              ▼
┌─────────────────────────────────────────────────────────────────┐
│                     HOOK: "setup"                               │
├─────────────────────────────────────────────────────────────────┤
│  - Les packs de compendiums sont indexés                        │
│  - Finaliser la configuration (trackable attributes, etc.)      │
│  - Les données du monde ne sont TOUJOURS PAS disponibles        │
└─────────────────────────────────────────────────────────────────┘
                              │
                              ▼
┌─────────────────────────────────────────────────────────────────┐
│                     HOOK: "ready"                               │
├─────────────────────────────────────────────────────────────────┤
│  - Le monde est entièrement chargé                              │
│  - game.actors, game.items, game.scenes sont disponibles        │
│  - Effectuer les migrations de données si nécessaire            │
│  - Initialiser les interfaces persistantes (HUD, etc.)          │
│  - Enregistrer les hooks d'interaction (hotbarDrop, etc.)       │
└─────────────────────────────────────────────────────────────────┘
```

## Exemple d'Initialisation (dnd5e)

Extrait du point d'entrée principal `dnd5e.mjs` :

```javascript
// Hook "init" - Configuration initiale
Hooks.once("init", function() {
  // Exposer le système globalement
  globalThis.dnd5e = game.dnd5e = Object.assign(game.system, globalThis.dnd5e);
  
  // Enregistrer la configuration
  CONFIG.DND5E = DND5E;
  
  // Enregistrer les classes de documents
  CONFIG.Actor.documentClass = documents.Actor5e;
  CONFIG.Item.documentClass = documents.Item5e;
  CONFIG.ActiveEffect.documentClass = documents.ActiveEffect5e;
  CONFIG.ChatMessage.documentClass = documents.ChatMessage5e;
  CONFIG.Combat.documentClass = documents.Combat5e;
  
  // Enregistrer les DataModels
  CONFIG.Actor.dataModels = dataModels.actor.config;
  CONFIG.Item.dataModels = dataModels.item.config;
  
  // Enregistrer les classes de dés
  CONFIG.Dice.D20Roll = dice.D20Roll;
  CONFIG.Dice.DamageRoll = dice.DamageRoll;
  
  // Enregistrer les feuilles
  DocumentSheetConfig.registerSheet(Actor, "dnd5e", CharacterActorSheet, {
    types: ["character"],
    makeDefault: true
  });
  
  // Enregistrer les paramètres
  registerSystemSettings();
  
  // Précharger les templates
  utils.preloadHandlebarsTemplates();
});

// Hook "setup" - Configuration avancée
Hooks.once("setup", function() {
  // Configurer les attributs trackables
  _configureTrackableAttributes();
  
  // Enregistrer l'art des modules
  game.dnd5e.moduleArt.registerModuleArt();
});

// Hook "ready" - Le monde est chargé
Hooks.once("ready", function() {
  // Vérifier et effectuer les migrations
  if ( game.user.isGM ) {
    migrations.migrateWorld();
  }
  
  // Enregistrer les hooks d'interaction
  Hooks.on("hotbarDrop", (bar, data, slot) => {
    if ( ["Item", "ActiveEffect"].includes(data.type) ) {
      documents.macro.create5eMacro(data, slot);
      return false;
    }
  });
});
```

## Flux de Données

```
┌─────────────┐     ┌──────────────┐     ┌─────────────┐
│   system    │────▶│  DataModel   │────▶│  Document   │
│    .json    │     │ (TypeData)   │     │  (Actor,    │
│             │     │              │     │   Item...)  │
└─────────────┘     └──────────────┘     └─────────────┘
       │                   │                    │
       │                   │                    │
       ▼                   ▼                    ▼
┌─────────────┐     ┌──────────────┐     ┌─────────────┐
│ documentTypes │    │ defineSchema │     │ prepareData │
│ packs        │    │ validation   │     │ _preCreate  │
│ compatibility│    │ migration    │     │ _onCreate   │
└─────────────┘     └──────────────┘     └─────────────┘
                                                │
                                                ▼
                                        ┌─────────────┐
                                        │ Application │
                                        │  (Sheet)    │
                                        │             │
                                        └─────────────┘
```

## Points Clés pour un Nouveau Système

1. **Commencer par `system.json`** - Définir les types de documents (Actor, Item)
2. **Créer les DataModels** - Schémas de validation avec `defineSchema()`
3. **Étendre les Documents** - Ajouter la logique métier dans `prepareData()`
4. **Créer les feuilles** - Interfaces utilisateur avec Application V2
5. **Tester les migrations** - Prévoir l'évolution du schéma de données

## Liens vers la Documentation Officielle

- [API FoundryVTT v13](https://foundryvtt.com/api/)
- [System Development Guide](https://foundryvtt.wiki/en/development/guides/SD-tutorial/SD01-Getting-started)
- [DataModel](https://foundryvtt.com/api/classes/foundry.abstract.DataModel.html)
- [TypeDataModel](https://foundryvtt.com/api/classes/foundry.abstract.TypeDataModel.html)

---

*Documentation suivante : [01-system-json.md](./01-system-json.md) - Guide complet du manifeste system.json*
