# Bonnes Pratiques pour Développer un Système FoundryVTT v13

**ID**: DND5E-PRACTICES-001  
**Date**: 2026-01-05  
**Basé sur**: Analyse du système dnd5e 5.2.0 et documentation officielle FoundryVTT v13

---

## 1. Architecture et Organisation du Code

### 1.1 Structure Modulaire avec Barrels

**Pattern recommandé** : Utiliser des fichiers `_module.mjs` (barrels) pour regrouper les exports par domaine.

```
module/
├── applications/
│   ├── actor/
│   │   ├── character-sheet.mjs
│   │   ├── npc-sheet.mjs
│   │   └── _module.mjs          # Export tous les sheets actor
│   ├── item/
│   │   └── _module.mjs
│   └── _module.mjs              # Agrège tous les sous-modules
├── data/
│   ├── actor/
│   │   └── _module.mjs
│   ├── item/
│   │   └── _module.mjs
│   └── _module.mjs
├── documents/
│   └── _module.mjs
└── config/
    ├── abilities.mjs
    ├── skills.mjs
    └── index.mjs                # Agrège la configuration
```

**Exemple de barrel** (`_module.mjs`) :
```javascript
// module/applications/actor/_module.mjs
export { default as CharacterActorSheet } from "./character-sheet.mjs";
export { default as NPCActorSheet } from "./npc-sheet.mjs";
export { default as VehicleActorSheet } from "./vehicle-sheet.mjs";
```

**Avantages** :
- Imports propres dans le fichier principal
- Facilite le tree-shaking
- Évite les imports circulaires

### 1.2 Point d'Entrée Principal Structuré

**Pattern recommandé** (basé sur `dnd5e.mjs`) :

```javascript
// system.mjs - Point d'entrée principal

// 1. Imports organisés par domaine
import CONFIG_SYSTEM from "./module/config/index.mjs";
import * as applications from "./module/applications/_module.mjs";
import * as dataModels from "./module/data/_module.mjs";
import * as documents from "./module/documents/_module.mjs";

// 2. Namespace global cohérent
globalThis.mysystem = {
  applications,
  config: CONFIG_SYSTEM,
  dataModels,
  documents,
  utils: {},
  ui: {}
};

// 3. Hooks bien ordonnés
Hooks.once("init", function() {
  // Configuration initiale
});

Hooks.once("setup", function() {
  // Configuration post-modules
});

Hooks.once("i18nInit", () => {
  // Pré-localisation
});

Hooks.once("ready", function() {
  // Migrations et initialisation finale
});
```

### 1.3 Configuration Découpée par Domaine

**Anti-pattern** : Un fichier `config.mjs` monolithique de 2000+ lignes.

**Pattern recommandé** : Découpage en fichiers thématiques.

```javascript
// module/config/abilities.mjs
export const abilities = {
  str: {
    label: "SYSTEM.AbilityStr",
    abbreviation: "SYSTEM.AbilityStrAbbr",
    type: "physical",
    icon: "systems/mysystem/icons/abilities/strength.svg"
  },
  dex: {
    label: "SYSTEM.AbilityDex",
    // ...
  }
};
```

```javascript
// module/config/index.mjs
import { abilities } from "./abilities.mjs";
import { skills } from "./skills.mjs";
import { weapons } from "./weapons.mjs";

const CONFIG_SYSTEM = {
  abilities,
  skills,
  weapons,
  // ...
};

export default CONFIG_SYSTEM;
```

---

## 2. Utilisation des APIs Modernes FoundryVTT v13

### 2.1 ApplicationV2 pour les Interfaces

**FoundryVTT v13** introduit `ApplicationV2` qui remplace l'ancienne API `Application`.

```javascript
// RECOMMANDÉ : ApplicationV2
import { ApplicationV2, HandlebarsApplicationMixin } from "foundry.applications.api";

class CharacterSheet extends HandlebarsApplicationMixin(ApplicationV2) {
  static DEFAULT_OPTIONS = {
    classes: ["mysystem", "sheet", "character"],
    position: { width: 600, height: 800 },
    window: {
      resizable: true
    },
    form: {
      handler: CharacterSheet.#onSubmit,
      submitOnChange: true
    }
  };

  static PARTS = {
    header: { template: "systems/mysystem/templates/actor/header.hbs" },
    tabs: { template: "systems/mysystem/templates/actor/tabs.hbs" },
    body: { template: "systems/mysystem/templates/actor/body.hbs" }
  };

  static async #onSubmit(event, form, formData) {
    await this.document.update(formData.object);
  }
}
```

### 2.2 DataModel pour les Données

**Pattern recommandé** : Utiliser `foundry.abstract.TypeDataModel` pour définir les schémas de données.

```javascript
// module/data/actor/character-data.mjs
export default class CharacterData extends foundry.abstract.TypeDataModel {
  static defineSchema() {
    const fields = foundry.data.fields;
    return {
      abilities: new fields.SchemaField({
        str: new fields.SchemaField({
          value: new fields.NumberField({ 
            required: true, 
            nullable: false, 
            integer: true, 
            initial: 10,
            min: 1, 
            max: 30 
          }),
          mod: new fields.NumberField({ integer: true })
        }),
        // ...autres abilities
      }),
      attributes: new fields.SchemaField({
        hp: new fields.SchemaField({
          value: new fields.NumberField({ integer: true, min: 0 }),
          max: new fields.NumberField({ integer: true, min: 0 }),
          temp: new fields.NumberField({ integer: true, min: 0, initial: 0 })
        })
      }),
      details: new fields.SchemaField({
        level: new fields.NumberField({ integer: true, min: 1, max: 20, initial: 1 }),
        xp: new fields.NumberField({ integer: true, min: 0, initial: 0 })
      })
    };
  }

  // Méthodes dérivées
  prepareDerivedData() {
    for (const [key, ability] of Object.entries(this.abilities)) {
      ability.mod = Math.floor((ability.value - 10) / 2);
    }
  }
}
```

### 2.3 Enregistrement des DataModels

```javascript
// Dans le hook init
Hooks.once("init", function() {
  CONFIG.Actor.dataModels = {
    character: CharacterData,
    npc: NPCData,
    vehicle: VehicleData
  };
  
  CONFIG.Item.dataModels = {
    weapon: WeaponData,
    armor: ArmorData,
    spell: SpellData
  };
});
```

---

## 3. Internationalisation (i18n)

### 3.1 Conventions de Clés

**Pattern recommandé** : Préfixe cohérent avec hiérarchie logique.

```json
{
  "MYSYSTEM": {
    "Ability": {
      "Str": "Force",
      "StrAbbr": "FOR",
      "Dex": "Dextérité",
      "DexAbbr": "DEX"
    },
    "Sheet": {
      "Character": "Feuille de Personnage",
      "NPC": "Feuille de PNJ"
    },
    "Action": {
      "Roll": "Lancer",
      "Save": "Sauvegarde"
    }
  }
}
```

### 3.2 Système de Pré-localisation

**Pattern copié de dnd5e** : Pré-localiser les configs au chargement pour éviter les lookups répétés.

```javascript
// module/utils/localization.mjs

const _preLocalizationRegistrations = {};

/**
 * Enregistre un chemin de config pour pré-localisation
 * @param {string} configKeyPath - Chemin dans CONFIG.MYSYSTEM (ex: "abilities")
 * @param {object} options
 * @param {string} [options.key] - Clé à localiser (ex: "label")
 * @param {string[]} [options.keys] - Clés multiples à localiser
 * @param {boolean} [options.sort] - Trier alphabétiquement après localisation
 */
export function preLocalize(configKeyPath, { key, keys = [], sort = false } = {}) {
  if (key) keys.unshift(key);
  _preLocalizationRegistrations[configKeyPath] = { keys, sort };
}

/**
 * Exécute toutes les pré-localisations enregistrées
 * @param {object} config - L'objet CONFIG.MYSYSTEM
 */
export function performPreLocalization(config) {
  for (const [keyPath, { keys, sort }] of Object.entries(_preLocalizationRegistrations)) {
    const target = foundry.utils.getProperty(config, keyPath);
    if (!target) continue;
    
    _localizeObject(target, keys, sort);
  }
}

function _localizeObject(obj, keys, sort) {
  for (const [k, v] of Object.entries(obj)) {
    if (typeof v === "object" && v !== null) {
      for (const key of keys) {
        if (typeof v[key] === "string") {
          v[key] = game.i18n.localize(v[key]);
        }
      }
    }
  }
  
  if (sort) {
    const sortedEntries = Object.entries(obj).sort((a, b) => 
      (a[1].label ?? "").localeCompare(b[1].label ?? "", game.i18n.lang)
    );
    for (const key of Object.keys(obj)) delete obj[key];
    for (const [key, value] of sortedEntries) obj[key] = value;
  }
}
```

**Utilisation** :

```javascript
// Dans config/index.mjs
import { preLocalize } from "../utils/localization.mjs";

preLocalize("abilities", { key: "label", sort: true });
preLocalize("skills", { keys: ["label", "abbreviation"], sort: true });
```

```javascript
// Dans le hook i18nInit
Hooks.once("i18nInit", () => {
  performPreLocalization(CONFIG.MYSYSTEM);
});
```

---

## 4. Enrichers de Texte Personnalisés

### 4.1 Pattern de Base

```javascript
// module/enrichers.mjs

export function registerCustomEnrichers() {
  CONFIG.TextEditor.enrichers.push({
    // Pattern : [[/roll 2d6+3]] ou [[/damage 1d8 fire]]
    pattern: /\[\[\/(?<type>roll|damage|check|save)(?<config> .*?)?]](?:{(?<label>[^}]+)})?/gi,
    enricher: enrichRoll
  });

  // Ajouter les listeners
  document.body.addEventListener("click", handleRollClick);
}

async function enrichRoll(match, options) {
  const { type, config, label } = match.groups;
  const parsedConfig = parseConfig(config);
  
  switch (type.toLowerCase()) {
    case "roll":
      return createRollLink(parsedConfig, label);
    case "damage":
      return createDamageLink(parsedConfig, label);
    case "check":
      return createCheckLink(parsedConfig, label);
    case "save":
      return createSaveLink(parsedConfig, label);
  }
  return null;
}

function parseConfig(configString = "") {
  const config = { values: [] };
  for (const part of configString.match(/(?:[^\s"]+|"[^"]*")+/g) ?? []) {
    if (!part) continue;
    const [key, value] = part.split("=");
    if (value === undefined) config.values.push(key);
    else if (value === "true" || value === "false") config[key] = value === "true";
    else if (Number.isNumeric(value)) config[key] = Number(value);
    else config[key] = value.replace(/(^"|"$)/g, "");
  }
  return config;
}

function createRollLink(config, label) {
  const formula = config.values.join(" ");
  const anchor = document.createElement("a");
  anchor.classList.add("roll-action");
  anchor.dataset.type = "roll";
  anchor.dataset.formula = formula;
  anchor.innerHTML = `<i class="fa-solid fa-dice-d20" inert></i> ${label || formula}`;
  return anchor;
}

async function handleRollClick(event) {
  const target = event.target.closest(".roll-action");
  if (!target) return;
  
  event.preventDefault();
  const { type, formula } = target.dataset;
  
  const roll = new Roll(formula);
  await roll.evaluate();
  await roll.toMessage({
    speaker: ChatMessage.getSpeaker()
  });
}
```

---

## 5. Système de Migrations

### 5.1 Architecture de Migration

```javascript
// module/migration.mjs

/**
 * Migration du monde entier
 */
export async function migrateWorld() {
  const version = game.system.version;
  const currentVersion = game.settings.get("mysystem", "systemMigrationVersion");
  
  if (!foundry.utils.isNewerVersion(version, currentVersion)) return;
  
  ui.notifications.info(`Migration vers la version ${version}...`, { permanent: true });
  
  // Migrer les acteurs
  for (const actor of game.actors) {
    try {
      const updateData = migrateActorData(actor.toObject());
      if (!foundry.utils.isEmpty(updateData)) {
        await actor.update(updateData, { enforceTypes: false });
      }
    } catch (err) {
      console.error(`Erreur de migration pour l'acteur ${actor.name}:`, err);
    }
  }
  
  // Migrer les items
  for (const item of game.items) {
    try {
      const updateData = migrateItemData(item.toObject());
      if (!foundry.utils.isEmpty(updateData)) {
        await item.update(updateData, { enforceTypes: false });
      }
    } catch (err) {
      console.error(`Erreur de migration pour l'item ${item.name}:`, err);
    }
  }
  
  // Marquer comme migré
  await game.settings.set("mysystem", "systemMigrationVersion", version);
  ui.notifications.info(`Migration vers ${version} terminée.`);
}

/**
 * Migrer les données d'un acteur
 */
function migrateActorData(actorData) {
  const updateData = {};
  
  // Exemple : Renommer un champ
  if (actorData.system?.attributes?.health !== undefined) {
    updateData["system.attributes.hp.value"] = actorData.system.attributes.health;
    updateData["system.attributes.-=health"] = null;
  }
  
  // Migrer les items embarqués
  if (actorData.items) {
    const items = actorData.items.map(i => {
      const itemUpdate = migrateItemData(i);
      return foundry.utils.mergeObject(i, itemUpdate, { enforceTypes: false });
    });
    if (items.some((i, idx) => i !== actorData.items[idx])) {
      updateData.items = items;
    }
  }
  
  return updateData;
}

function migrateItemData(itemData) {
  const updateData = {};
  
  // Migrations spécifiques aux items
  
  return updateData;
}
```

### 5.2 Enregistrement du Setting de Version

```javascript
// module/settings.mjs
export function registerSystemSettings() {
  game.settings.register("mysystem", "systemMigrationVersion", {
    name: "Version de Migration",
    scope: "world",
    config: false,
    type: String,
    default: ""
  });
}
```

---

## 6. Registres et Gestion des Relations

### 6.1 Pattern Registry

```javascript
// module/registry.mjs

/**
 * Registre générique pour tracker les documents par type
 */
class ItemRegistry {
  constructor(type) {
    this.type = type;
    this._items = new Map();
  }

  initialize() {
    this._items.clear();
    for (const item of game.items) {
      if (item.type === this.type) {
        this._items.set(item.id, item);
      }
    }
  }

  get(id) {
    return this._items.get(id);
  }

  getAll() {
    return Array.from(this._items.values());
  }

  register(item) {
    if (item.type !== this.type) return;
    this._items.set(item.id, item);
  }

  unregister(id) {
    this._items.delete(id);
  }
}

export default {
  classes: new ItemRegistry("class"),
  spells: new ItemRegistry("spell"),
  // ...
};
```

---

## 7. Performance

### 7.1 Lazy Loading

```javascript
// Charger les modules lourds uniquement quand nécessaire
async function openAdvancedConfig() {
  const { AdvancedConfigApp } = await import("./applications/advanced-config.mjs");
  new AdvancedConfigApp().render(true);
}
```

### 7.2 Debounce des Updates Fréquents

```javascript
// Pour les champs qui changent souvent (ex: sliders)
static DEFAULT_OPTIONS = {
  form: {
    handler: this.#onSubmit,
    submitOnChange: true,
    submitOnClose: true,
    closeOnSubmit: false
  }
};

// Utiliser debounce dans les handlers
_onInputChange = foundry.utils.debounce((event) => {
  this._updateObject(event, new FormDataExtended(this.element));
}, 300);
```

### 7.3 Préparation des Données Efficace

```javascript
// Éviter les calculs redondants
prepareDerivedData() {
  // Cacher les valeurs calculées une seule fois
  this._cachedModifiers = null;
}

get modifiers() {
  if (this._cachedModifiers === null) {
    this._cachedModifiers = this._computeModifiers();
  }
  return this._cachedModifiers;
}
```

---

## 8. Accessibilité

### 8.1 Attributs ARIA

```html
<!-- Templates accessibles -->
<button type="button" 
        aria-label="{{localize 'MYSYSTEM.Action.Roll'}}"
        aria-describedby="roll-description">
  <i class="fas fa-dice-d20" aria-hidden="true"></i>
</button>
<span id="roll-description" class="sr-only">
  {{localize 'MYSYSTEM.Action.RollDescription'}}
</span>
```

### 8.2 Navigation Clavier

```javascript
// Supporter les actions clavier
element.addEventListener("keydown", (event) => {
  if (event.key === "Enter" || event.key === " ") {
    event.preventDefault();
    this._onRollClick(event);
  }
});
```

---

## 9. Tests et Qualité

### 9.1 Structure Recommandée pour les Tests

```
tests/
├── unit/
│   ├── data/
│   │   └── character-data.test.mjs
│   ├── utils/
│   │   └── calculations.test.mjs
│   └── setup.mjs
├── integration/
│   └── actor-creation.test.mjs
└── README.md
```

### 9.2 Validation des Données avec DataModel

```javascript
// Les DataModels valident automatiquement
static defineSchema() {
  return {
    level: new fields.NumberField({
      required: true,
      nullable: false,
      integer: true,
      min: 1,
      max: 20,
      initial: 1,
      validate: (value) => {
        if (value < 1 || value > 20) {
          throw new Error("Le niveau doit être entre 1 et 20");
        }
      }
    })
  };
}
```

---

## 10. Checklist de Développement

### Avant de Commencer
- [ ] Définir la structure des dossiers
- [ ] Configurer le `system.json` avec les bonnes métadonnées
- [ ] Mettre en place les DataModels pour chaque type de document

### Pendant le Développement
- [ ] Utiliser ApplicationV2 pour les nouvelles interfaces
- [ ] Préfixer toutes les clés i18n avec le nom du système
- [ ] Documenter les hooks personnalisés
- [ ] Implémenter les migrations dès la première version

### Avant la Release
- [ ] Tester les migrations sur des données existantes
- [ ] Vérifier l'accessibilité (navigation clavier, contrastes)
- [ ] Valider les performances avec un monde chargé
- [ ] Documenter l'API publique pour les modules tiers

---

## Références

- [API FoundryVTT v13](https://foundryvtt.com/api/)
- [Discord Foundry - Canal Development](https://discord.gg/foundryvtt)
- [Système dnd5e - Référence architecturale](https://github.com/foundryvtt/dnd5e)

