# Anti-Patterns à Éviter pour un Système FoundryVTT v13

**ID**: DND5E-ANTIPATTERNS-001  
**Date**: 2026-01-05  
**Basé sur**: Analyse critique du système dnd5e 5.2.0

---

## Introduction

Ce document liste les **patterns à NE PAS reproduire** lors du développement d'un système FoundryVTT v13. Chaque anti-pattern est illustré par un exemple concret tiré du système dnd5e, suivi de l'alternative recommandée.

> ⚠️ **Important** : Le système dnd5e est une excellente référence architecturale, mais son historique long (depuis Foundry v0.x) a créé une dette technique significative. Ne pas reproduire aveuglément son code.

---

## 1. Fichiers Monolithiques

### ❌ Anti-Pattern : Configuration Géante

**Problème observé dans dnd5e** : Le fichier `config.mjs` fait plus de 2000 lignes.

```javascript
// ❌ À ÉVITER : Un seul fichier de configuration géant
// dnd5e/module/config.mjs - 2000+ lignes

const DND5E = {};

DND5E.abilities = {
  str: { label: "DND5E.AbilityStr", /* ... */ },
  dex: { label: "DND5E.AbilityDex", /* ... */ },
  // ...50+ lignes
};

DND5E.skills = {
  // ...100+ lignes
};

DND5E.weaponTypes = {
  // ...150+ lignes
};

DND5E.spellSchools = {
  // ...etc, 2000+ lignes au total
};

export default DND5E;
```

**Impacts** :
- Difficile à maintenir et naviguer
- Merge conflicts fréquents en équipe
- Temps de chargement initial plus long
- Impossible de lazy-load des portions

### ✅ Pattern Recommandé : Configuration Découpée

```javascript
// ✅ RECOMMANDÉ : Fichiers séparés par domaine

// config/abilities.mjs
export const abilities = {
  str: { label: "SYSTEM.AbilityStr", abbreviation: "SYSTEM.AbilityStrAbbr" },
  dex: { label: "SYSTEM.AbilityDex", abbreviation: "SYSTEM.AbilityDexAbbr" },
  // ...
};

// config/skills.mjs
export const skills = {
  acrobatics: { label: "SYSTEM.SkillAcr", ability: "dex" },
  // ...
};

// config/index.mjs - Agrégation
import { abilities } from "./abilities.mjs";
import { skills } from "./skills.mjs";
import { weapons } from "./weapons.mjs";

const CONFIG_SYSTEM = { abilities, skills, weapons };
export default CONFIG_SYSTEM;
```

**Avantages** :
- Fichiers de 50-200 lignes maximum
- Responsabilité unique par fichier
- Facilite les revues de code
- Possibilité de lazy-loading

---

## 2. Fichiers Utilitaires Fourre-Tout

### ❌ Anti-Pattern : Utils Géant

**Problème observé** : `utils.mjs` contient 1390 lignes de fonctions très diverses.

```javascript
// ❌ À ÉVITER : utils.mjs fourre-tout (1390 lignes dans dnd5e)

// Formatage de nombres
export function formatNumber(value, options) { /* ... */ }
export function formatCR(value) { /* ... */ }
export function formatModifier(value) { /* ... */ }

// Conversion d'unités
export function convertDistance(value, from, to) { /* ... */ }
export function convertWeight(value, from, to) { /* ... */ }

// Helpers Handlebars
export function registerHandlebarsHelpers() { /* ... */ }

// Localisation
export function preLocalize(path, options) { /* ... */ }
export function performPreLocalization(config) { /* ... */ }

// Validation
export function validateDice(formula) { /* ... */ }

// IDs statiques
export function staticID(identifier) { /* ... */ }

// Et 50+ autres fonctions sans rapport...
```

### ✅ Pattern Recommandé : Modules Spécialisés

```javascript
// ✅ RECOMMANDÉ : Séparation par responsabilité

// utils/formatting.mjs
export function formatNumber(value, options) { /* ... */ }
export function formatModifier(value) { /* ... */ }
export function formatCurrency(value, denomination) { /* ... */ }

// utils/conversion.mjs
export function convertDistance(value, from, to) { /* ... */ }
export function convertWeight(value, from, to) { /* ... */ }

// utils/localization.mjs
export function preLocalize(path, options) { /* ... */ }
export function performPreLocalization(config) { /* ... */ }

// utils/validation.mjs
export function validateDice(formula) { /* ... */ }
export function validateAbilityScore(value) { /* ... */ }

// utils/handlebars.mjs
export function registerHandlebarsHelpers() { /* ... */ }

// utils/index.mjs - Export centralisé si nécessaire
export * from "./formatting.mjs";
export * from "./conversion.mjs";
// ...
```

---

## 3. IDs de Compendium Hardcodés

### ❌ Anti-Pattern : UUIDs en Dur

**Problème observé dans dnd5e** : Les IDs de compendium sont hardcodés partout.

```javascript
// ❌ À ÉVITER : UUIDs hardcodés dans config.mjs

DND5E.weaponIds = {
  battleaxe: "Compendium.dnd5e.equipment24.Item.phbwepBattleaxe0",
  handaxe: "Compendium.dnd5e.equipment24.Item.phbwepHandaxe00",
  javelin: "Compendium.dnd5e.equipment24.Item.phbwepJavelin00",
  // ... des dizaines d'IDs hardcodés
};

DND5E.armorIds = {
  leather: "Compendium.dnd5e.equipment24.Item.phbarmLeather000",
  // ...
};

// Utilisation problématique ailleurs dans le code
const weapon = await fromUuid(CONFIG.DND5E.weaponIds.battleaxe);
```

**Impacts** :
- Impossible de surcharger sans modifier le système
- Si les IDs changent (migration), tout casse
- Couplage fort entre code et données
- Les modules ne peuvent pas substituer leurs propres items

### ✅ Pattern Recommandé : Lookup par Identifiant Métier

```javascript
// ✅ RECOMMANDÉ : Système de lookup flexible

// config/items.mjs
export const itemIdentifiers = {
  weapons: {
    battleaxe: { identifier: "battleaxe", pack: "equipment" },
    handaxe: { identifier: "handaxe", pack: "equipment" },
  },
  armor: {
    leather: { identifier: "leather", pack: "equipment" },
  }
};

// utils/item-lookup.mjs
const _itemCache = new Map();

/**
 * Récupère un item par son identifiant métier, pas par UUID
 */
export async function getItemByIdentifier(category, identifier) {
  const cacheKey = `${category}.${identifier}`;
  if (_itemCache.has(cacheKey)) return _itemCache.get(cacheKey);
  
  const config = CONFIG.SYSTEM.itemIdentifiers[category]?.[identifier];
  if (!config) return null;
  
  const pack = game.packs.get(`mysystem.${config.pack}`);
  if (!pack) return null;
  
  // Recherche par identifiant métier, pas par ID technique
  const index = await pack.getIndex({ fields: ["system.identifier"] });
  const entry = index.find(i => i.system?.identifier === config.identifier);
  
  if (entry) {
    const item = await pack.getDocument(entry._id);
    _itemCache.set(cacheKey, item);
    return item;
  }
  
  return null;
}

// Utilisation
const battleaxe = await getItemByIdentifier("weapons", "battleaxe");
```

**Avantages** :
- Les modules peuvent enregistrer des items alternatifs
- Les IDs peuvent changer sans casser le code
- Cache pour la performance
- Découplage code/données

---

## 4. Gestion Multi-Versions des Règles

### ❌ Anti-Pattern : Conditions Partout

**Problème observé** : dnd5e gère les règles 2014 et 2024 avec des conditions dispersées.

```javascript
// ❌ À ÉVITER : Vérifications de version éparpillées dans le code

// Dans config.mjs
if (game.settings.get("dnd5e", "rulesVersion") === "legacy") {
  applyLegacyRules();
}

// Dans un document
prepareData() {
  if (game.settings.get("dnd5e", "rulesVersion") === "legacy") {
    this.system.speed = this._computeLegacySpeed();
  } else {
    this.system.speed = this._computeModernSpeed();
  }
}

// Dans un template
{{#if isLegacy}}
  <span>{{legacyLabel}}</span>
{{else}}
  <span>{{modernLabel}}</span>
{{/if}}

// Dans les enrichers
export function getRulesVersion(config = {}, options = {}) {
  if (Number.isNumeric(config.rules)) return String(config.rules);
  return options.relativeTo?.parent?.system?.source?.rules
    || options.relativeTo?.system?.source?.rules
    || (game.settings.get("dnd5e", "rulesVersion") === "modern" ? "2024" : "2014");
}
```

**Impacts** :
- Complexité cognitive très élevée
- Double maintenance pour chaque fonctionnalité
- Bugs difficiles à tracer (quelle version ?)
- Tests exponentiellement plus complexes

### ✅ Pattern Recommandé : Choix Unique ou Modules

**Option A : Une seule version des règles**
```javascript
// ✅ RECOMMANDÉ : Choisir une version et s'y tenir
// Pas de conditions, pas de complexité

const CONFIG_SYSTEM = {
  // Règles 2024 uniquement
  speed: {
    walk: { label: "SYSTEM.Speed.Walk" },
    fly: { label: "SYSTEM.Speed.Fly" },
    // ...
  }
};
```

**Option B : Variantes via modules séparés**
```javascript
// ✅ RECOMMANDÉ : Un module séparé pour les variantes

// Système principal : règles modernes
// system.json
{
  "id": "mysystem",
  "version": "1.0.0"
}

// Module optionnel : règles legacy
// mysystem-legacy/module.json
{
  "id": "mysystem-legacy",
  "relationships": {
    "requires": [{ "id": "mysystem", "type": "system" }]
  }
}

// Le module surcharge proprement les configs nécessaires
Hooks.once("init", () => {
  foundry.utils.mergeObject(CONFIG.MYSYSTEM.speed, {
    walk: { label: "SYSTEM.Legacy.Speed.Walk" }
  });
});
```

---

## 5. APIs Dépréciées

### ❌ Anti-Pattern : Application v1

**Problème observé** : Utilisation de l'ancienne API Application.

```javascript
// ❌ À ÉVITER : Ancienne API Application (v1)

// Dans dnd5e.mjs ligne 139
DocumentSheetConfig.unregisterSheet(Actor, "core", foundry.appv1.sheets.ActorSheet);

// Sheets utilisant l'ancienne API
class CharacterSheet extends ActorSheet {
  static get defaultOptions() {
    return mergeObject(super.defaultOptions, {
      classes: ["dnd5e", "sheet", "actor"],
      width: 720,
      height: 680
    });
  }

  getData() {
    const data = super.getData();
    // ...
    return data;
  }

  activateListeners(html) {
    super.activateListeners(html);
    html.find(".rollable").click(this._onRoll.bind(this));
  }
}
```

### ✅ Pattern Recommandé : ApplicationV2

```javascript
// ✅ RECOMMANDÉ : Nouvelle API ApplicationV2

import { ApplicationV2, HandlebarsApplicationMixin } from "foundry.applications.api";

class CharacterSheet extends HandlebarsApplicationMixin(ApplicationV2) {
  static DEFAULT_OPTIONS = {
    classes: ["mysystem", "sheet", "actor", "character"],
    position: { width: 720, height: 680 },
    window: {
      resizable: true,
      controls: [
        { icon: "fas fa-cog", label: "SYSTEM.Settings", action: "openSettings" }
      ]
    },
    actions: {
      roll: CharacterSheet.#onRoll,
      openSettings: CharacterSheet.#openSettings
    }
  };

  static PARTS = {
    header: { template: "systems/mysystem/templates/actor/header.hbs" },
    navigation: { template: "systems/mysystem/templates/actor/navigation.hbs" },
    body: { template: "systems/mysystem/templates/actor/body.hbs" }
  };

  async _prepareContext(options) {
    const context = await super._prepareContext(options);
    context.actor = this.actor;
    context.system = this.actor.system;
    return context;
  }

  static async #onRoll(event, target) {
    const rollType = target.dataset.rollType;
    // Logique de roll...
  }

  static #openSettings(event, target) {
    // Ouvrir les settings...
  }
}
```

**Différences clés** :
- `DEFAULT_OPTIONS` au lieu de `defaultOptions()`
- `PARTS` pour les templates multi-sections
- `actions` déclaratives au lieu de `activateListeners()`
- `_prepareContext()` au lieu de `getData()`

---

## 6. Fonctions Trop Complexes

### ❌ Anti-Pattern : Fonctions Géantes

**Problème observé** : Certaines fonctions dépassent 150 lignes avec une complexité cyclomatique élevée.

```javascript
// ❌ À ÉVITER : Fonction géante (exemple simplifié de enrichCheck dans enrichers.mjs)

async function enrichCheck(config, label, options) {
  // 150+ lignes avec 15+ conditions imbriquées
  
  if (config.ability) {
    if (config.skill) {
      if (config.dc) {
        if (config.passive) {
          // ...
        } else {
          // ...
        }
      } else {
        // ...
      }
    } else if (config.tool) {
      // ... encore plus de conditions
    }
  } else if (config.values.length > 0) {
    for (const value of config.values) {
      if (value in CONFIG.DND5E.abilities) {
        // ...
      } else if (value in CONFIG.DND5E.skills) {
        // ...
      } else if (value in CONFIG.DND5E.tools) {
        // ...
      }
      // ... etc.
    }
  }
  
  // ... 100 lignes de plus
}
```

### ✅ Pattern Recommandé : Décomposition

```javascript
// ✅ RECOMMANDÉ : Fonctions courtes et spécialisées

async function enrichCheck(config, label, options) {
  const checkType = determineCheckType(config);
  const checkConfig = buildCheckConfig(config, checkType, options);
  
  if (!checkConfig) {
    console.warn(`Configuration de check invalide: ${config._input}`);
    return null;
  }
  
  return createCheckElement(checkConfig, label);
}

function determineCheckType(config) {
  if (config.skill) return "skill";
  if (config.tool) return "tool";
  if (config.ability) return "ability";
  
  // Déduction depuis les valeurs
  for (const value of config.values) {
    if (value in CONFIG.SYSTEM.skills) return "skill";
    if (value in CONFIG.SYSTEM.tools) return "tool";
    if (value in CONFIG.SYSTEM.abilities) return "ability";
  }
  
  return null;
}

function buildCheckConfig(config, type, options) {
  switch (type) {
    case "skill": return buildSkillCheckConfig(config, options);
    case "tool": return buildToolCheckConfig(config, options);
    case "ability": return buildAbilityCheckConfig(config, options);
    default: return null;
  }
}

function buildSkillCheckConfig(config, options) {
  return {
    type: "skill",
    skill: config.skill || config.values.find(v => v in CONFIG.SYSTEM.skills),
    ability: config.ability,
    dc: config.dc,
    passive: config.passive ?? false
  };
}

function createCheckElement(config, label) {
  const anchor = document.createElement("a");
  anchor.classList.add("roll-action", "check-roll");
  anchor.dataset.type = config.type;
  anchor.dataset.key = config.skill || config.tool || config.ability;
  if (config.dc) anchor.dataset.dc = config.dc;
  anchor.innerHTML = buildCheckLabel(config, label);
  return anchor;
}
```

**Règles** :
- Maximum 50-70 lignes par fonction
- Maximum 3-4 niveaux d'indentation
- Une seule responsabilité par fonction
- Noms descriptifs et intention claire

---

## 7. Absence de Types

### ❌ Anti-Pattern : JSDoc Insuffisant

**Problème observé** : Types documentés mais pas vérifiés.

```javascript
// ❌ À ÉVITER : Types en commentaires non vérifiés

/**
 * @import { AbilityConfiguration } from "./_types.mjs"
 */

/**
 * @param {string} ability 
 * @param {object} options
 * @returns {number}
 */
function calculateModifier(ability, options) {
  // Rien ne vérifie que 'ability' est bien une string
  // ou que le retour est bien un number
  return Math.floor((ability.value - 10) / 2); // Bug si ability est une string!
}
```

### ✅ Pattern Recommandé : TypeScript ou Validation Runtime

**Option A : TypeScript**
```typescript
// ✅ RECOMMANDÉ : TypeScript pour la vérification statique

interface AbilityScore {
  value: number;
  mod?: number;
}

interface Abilities {
  str: AbilityScore;
  dex: AbilityScore;
  con: AbilityScore;
  int: AbilityScore;
  wis: AbilityScore;
  cha: AbilityScore;
}

function calculateModifier(abilityScore: AbilityScore): number {
  return Math.floor((abilityScore.value - 10) / 2);
}
```

**Option B : Validation avec DataModel**
```javascript
// ✅ RECOMMANDÉ : Validation via DataModel de Foundry

class AbilityData extends foundry.abstract.DataModel {
  static defineSchema() {
    return {
      value: new foundry.data.fields.NumberField({
        required: true,
        nullable: false,
        integer: true,
        min: 1,
        max: 30,
        initial: 10
      })
    };
  }

  get mod() {
    return Math.floor((this.value - 10) / 2);
  }
}
```

---

## 8. Couplage UI/Logique Métier

### ❌ Anti-Pattern : Logique dans les Sheets

```javascript
// ❌ À ÉVITER : Calculs métier dans les applications UI

class CharacterSheet extends ApplicationV2 {
  async _prepareContext() {
    const context = await super._prepareContext();
    
    // ❌ Calcul de l'AC directement dans le sheet
    let ac = 10;
    ac += this.actor.system.abilities.dex.mod;
    const armor = this.actor.items.find(i => i.type === "armor" && i.system.equipped);
    if (armor) {
      ac = armor.system.ac.base;
      if (armor.system.ac.dexLimit !== null) {
        ac += Math.min(this.actor.system.abilities.dex.mod, armor.system.ac.dexLimit);
      }
    }
    context.ac = ac;
    
    return context;
  }
}
```

### ✅ Pattern Recommandé : Séparation des Responsabilités

```javascript
// ✅ RECOMMANDÉ : Logique métier dans le Document/DataModel

// data/actor/character-data.mjs
class CharacterData extends foundry.abstract.TypeDataModel {
  prepareDerivedData() {
    this.attributes.ac.value = this._computeArmorClass();
  }

  _computeArmorClass() {
    const dexMod = this.abilities.dex.mod;
    const armor = this.parent.items.find(i => 
      i.type === "armor" && i.system.equipped
    );
    
    if (!armor) return 10 + dexMod;
    
    const baseAC = armor.system.ac.base;
    const dexLimit = armor.system.ac.dexLimit;
    const dexBonus = dexLimit !== null ? Math.min(dexMod, dexLimit) : dexMod;
    
    return baseAC + dexBonus;
  }
}

// applications/actor/character-sheet.mjs
class CharacterSheet extends ApplicationV2 {
  async _prepareContext() {
    const context = await super._prepareContext();
    // ✅ Le sheet utilise simplement la valeur calculée
    context.ac = this.actor.system.attributes.ac.value;
    return context;
  }
}
```

---

## 9. Absence de Tests

### ❌ Anti-Pattern : Pas de Tests Automatisés

**Problème observé** : Le repository dnd5e analysé ne contient pas de tests automatisés visibles.

```javascript
// ❌ À ÉVITER : Code critique sans tests

// Migration sans tests
export async function migrateWorld() {
  // 180 lignes de code critique
  // Si ça casse, les données des utilisateurs sont corrompues
  // Aucun test automatisé pour vérifier
}

// Calculs métier sans tests
function calculateEncumbrance(actor) {
  // Logique complexe
  // Aucune vérification que les edge cases fonctionnent
}
```

### ✅ Pattern Recommandé : Tests Systématiques

```javascript
// ✅ RECOMMANDÉ : Tests pour le code critique

// tests/unit/calculations.test.mjs
import { calculateModifier, calculateEncumbrance } from "../../module/utils/calculations.mjs";

describe("calculateModifier", () => {
  it("should return -1 for score of 8", () => {
    expect(calculateModifier(8)).toBe(-1);
  });

  it("should return 0 for score of 10", () => {
    expect(calculateModifier(10)).toBe(0);
  });

  it("should return +5 for score of 20", () => {
    expect(calculateModifier(20)).toBe(5);
  });

  it("should handle edge case of 1", () => {
    expect(calculateModifier(1)).toBe(-5);
  });
});

// tests/unit/migration.test.mjs
describe("migrateActorData", () => {
  it("should migrate old health field to hp", () => {
    const oldData = {
      system: { attributes: { health: 45 } }
    };
    const result = migrateActorData(oldData);
    expect(result["system.attributes.hp.value"]).toBe(45);
    expect(result["system.attributes.-=health"]).toBeNull();
  });

  it("should not modify actors without old fields", () => {
    const newData = {
      system: { attributes: { hp: { value: 45 } } }
    };
    const result = migrateActorData(newData);
    expect(Object.keys(result)).toHaveLength(0);
  });
});
```

---

## 10. Résumé : Checklist Anti-Patterns

### À Éviter Absolument

| Anti-Pattern | Impact | Alternative |
|--------------|--------|-------------|
| Fichier config 2000+ lignes | Maintenance impossible | Découper par domaine |
| Utils fourre-tout | Code spaghetti | Modules spécialisés |
| UUIDs hardcodés | Couplage fort | Lookup par identifiant métier |
| Multi-versions inline | Complexité x2 | Un seul set de règles ou modules |
| Application v1 | API dépréciée | ApplicationV2 |
| Fonctions 150+ lignes | Bugs difficiles | Décomposition |
| Pas de types | Erreurs runtime | TypeScript ou DataModel |
| Logique dans UI | Couplage | Séparation document/sheet |
| Pas de tests | Régressions | Tests unitaires |

### Questions à Se Poser

Avant de valider du code, vérifier :

1. **Ce fichier fait-il plus de 300 lignes ?** → Peut-on le découper ?
2. **Cette fonction fait-elle plus de 50 lignes ?** → Peut-on la décomposer ?
3. **Y a-t-il des IDs/UUIDs en dur ?** → Peut-on utiliser un système de lookup ?
4. **Y a-t-il des conditions de version ?** → Peut-on simplifier ?
5. **Cette logique est-elle dans un sheet ?** → Devrait-elle être dans le document ?
6. **Ce code critique a-t-il des tests ?** → Ajouter des tests unitaires.

---

## Références

- [Analyse Critique dnd5e](./analyse-critique-dnd5e.md)
- [Bonnes Pratiques](./bonnes-pratiques.md)
- [API FoundryVTT v13](https://foundryvtt.com/api/)
