# Analyse Critique du Système dnd5e pour FoundryVTT v13

**ID**: DND5E-ANALYSE-001  
**Version analysée**: 5.2.0 (compatible FoundryVTT v13.347+)  
**Date d'analyse**: 2026-01-05  

---

## 1. Vue d'ensemble

Le système dnd5e est le système officiel de Foundry VTT, développé par Atropos. C'est une référence en termes de patterns et d'architecture, mais son historique long (depuis Foundry v0.x) implique une **dette technique significative** et des patterns legacy qui ne doivent pas être reproduits dans un nouveau système v13.

---

## 2. Points Forts (À Reproduire)

### 2.1 Architecture Modulaire

**Fichier principal** : `dnd5e.mjs`

Le point d'entrée est exemplaire dans sa structure :

```javascript
// Import bien organisé par domaines
import * as applications from "./module/applications/_module.mjs";
import * as canvas from "./module/canvas/_module.mjs";
import * as dataModels from "./module/data/_module.mjs";
import * as dice from "./module/dice/_module.mjs";
import * as documents from "./module/documents/_module.mjs";
```

**Points positifs** :
- Séparation claire des responsabilités (applications, data, documents, canvas)
- Utilisation de modules barrel (`_module.mjs`) pour regrouper les exports
- Namespace global cohérent via `globalThis.dnd5e`

### 2.2 Système de Configuration Centralisé

**Fichier** : `module/config.mjs`

Le fichier `config.mjs` (2000+ lignes) centralise toute la configuration métier :

```javascript
DND5E.abilities = {
  str: {
    label: "DND5E.AbilityStr",
    abbreviation: "DND5E.AbilityStrAbbr",
    type: "physical",
    fullKey: "strength",
    reference: "Compendium.dnd5e.content24.JournalEntry...",
    icon: "systems/dnd5e/icons/svg/abilities/strength.svg"
  },
  // ...
};
```

**Points positifs** :
- Configuration déclarative et extensible
- Clés i18n cohérentes (pattern `DND5E.XXX`)
- Références vers le contenu (compendiums)
- Icônes associées aux concepts

### 2.3 Système de Pré-localisation

**Fichier** : `module/utils.mjs` (fonction `preLocalize`)

```javascript
export function preLocalize(configKeyPath, { key, keys=[], sort=false }={}) {
  if ( key ) keys.unshift(key);
  _preLocalizationRegistrations[configKeyPath] = { keys, sort };
}
```

**Avantages** :
- Localisation automatique des configs au chargement
- Tri alphabétique optionnel selon la langue
- Performance optimisée (une seule passe)

### 2.4 Système de Registres

**Fichier** : `module/registry.mjs`

Architecture de registres pour gérer les relations complexes :

```javascript
export default {
  classes: new ItemRegistry("class"),
  dependents: DependentsRegistry,
  enchantments: EnchantmentRegisty,
  messages: MessageRegistry,
  spellLists: SpellListRegistry,
  summons: SummonRegistry
};
```

**Points positifs** :
- Gestion des dépendances entre documents
- Tracking des enchantements et invocations
- Support Babele (traduction des compendiums)

### 2.5 Enrichers de Texte Personnalisés

**Fichier** : `module/enrichers.mjs`

Système riche d'enrichissement de texte :

```javascript
const stringNames = [
  "attack", "award", "check", "concentration", "damage", "heal", "healing", "item", "save", "skill", "tool"
];
CONFIG.TextEditor.enrichers.push({
  pattern: new RegExp(`\\[\\[/(?<type>${stringNames.join("|")})(?<config> .*?)?]]...`),
  enricher: enrichString
});
```

**Exemples de syntaxe** :
- `[[/attack +5]]` - Lien de jet d'attaque
- `[[/save ability=dex dc=15]]` - Jet de sauvegarde
- `[[/damage 2d6 fire]]` - Jet de dégâts

### 2.6 Gestion des Migrations

**Fichier** : `module/migration.mjs`

Système de migration robuste avec :
- Versionnage des données (`systemMigrationVersion`)
- Migration par type de document
- Gestion des compendiums world vs system
- Progression visuelle

```javascript
export async function migrateWorld({ bypassVersionCheck=false }={}) {
  const version = game.system.version;
  // Progress notification
  const progress = ui.notifications.info("MIGRATION.5eBegin", { ... });
  // ...
}
```

### 2.7 Hook System Bien Utilisé

Utilisation cohérente des hooks Foundry :

```javascript
Hooks.once("init", function() { /* Configuration initiale */ });
Hooks.once("setup", function() { /* Configuration des modules */ });
Hooks.once("i18nInit", () => { /* Localisation */ });
Hooks.once("ready", function() { /* Migration et initialisation finale */ });
```

---

## 3. Points Faibles (À Éviter)

### 3.1 Fichier de Configuration Monolithique

**Problème** : `config.mjs` fait 2000+ lignes

**Impact** :
- Difficile à maintenir
- Pas de séparation par domaine métier
- Merge conflicts fréquents en équipe

**Recommandation** : Découper en fichiers par domaine :
```
config/
  abilities.mjs
  skills.mjs
  weapons.mjs
  spells.mjs
  index.mjs  # Agrégation
```

### 3.2 Duplication Legacy/Modern

**Problème** : Le système gère deux versions des règles (2014/2024) avec duplication :

```javascript
// Dans config.mjs
if ( game.settings.get("dnd5e", "rulesVersion") === "legacy" ) applyLegacyRules();
```

**Impact** :
- Code conditionnel partout
- Double maintenance
- Complexité cognitive élevée

**Pour un nouveau système** : Choisir une version des règles et s'y tenir, ou utiliser un système de modules pour les variantes.

### 3.3 Couplage Fort avec les Compendiums

**Problème** : Les IDs de compendium sont hardcodés :

```javascript
DND5E.weaponIds = {
  battleaxe: "Compendium.dnd5e.equipment24.Item.phbwepBattleaxe0",
  // ...
};
```

**Impact** :
- Impossible de surcharger sans conflit
- Dépendance forte aux données
- Problèmes de migration si les IDs changent

**Recommandation** : Utiliser un système de lookup par identifiant métier plutôt que par UUID.

### 3.4 Absence de Types TypeScript

**Problème** : Utilisation de JSDoc au lieu de TypeScript

```javascript
/**
 * @import { AbilityConfiguration, ... } from "./_types.mjs";
 */
```

**Impact** :
- Pas de vérification de types au build
- Documentation manuelle des types
- Erreurs runtime potentielles

**Pour v13** : Envisager TypeScript ou au minimum des fichiers `.d.ts` pour les types publics.

### 3.5 Gestion Complexe des Activités

**Structure** : `module/data/activity/` et `module/documents/activity/`

Le système d'activités (attack, damage, heal, etc.) est puissant mais **très complexe** :
- 14+ types d'activités
- Mixins et héritage complexe
- Couplage fort avec les items

**Pour un nouveau système** : Simplifier si possible, ou documenter extensivement.

### 3.6 Utils Fourre-Tout

**Fichier** : `module/utils.mjs` (1390 lignes)

Contient des fonctions très diverses :
- Formatage (formatNumber, formatCR, formatModifier...)
- Conversion d'unités
- Helpers Handlebars
- Localisation
- Validation

**Recommandation** : Séparer en modules spécialisés :
```
utils/
  formatting.mjs
  conversion.mjs
  handlebars.mjs
  localization.mjs
```

---

## 4. Dette Technique Identifiée

### 4.1 Patterns Dépréciés

| Pattern | Fichier | Recommandation v13 |
|---------|---------|-------------------|
| `foundry.appv1.sheets` | `dnd5e.mjs:139` | Utiliser ApplicationV2 |
| `CONFIG.ui.chat` custom | `dnd5e.mjs:90` | Vérifier alternatives v13 |
| `KeyboardManager.MODIFIER_CODES` | `utils.mjs:430` | Utiliser nouvelle API |

### 4.2 Complexité Cyclomatique

Certaines fonctions dépassent les seuils recommandés :
- `enrichCheck()` : ~150 lignes, 15+ conditions
- `migrateWorld()` : ~180 lignes
- `_configureStatusEffects()` : logique imbriquée

### 4.3 Manque de Tests Unitaires

Pas de tests automatisés visibles dans le repository analysé. Les migrations et validations métier sont critiques et devraient être testées.

---

## 5. Analyse du Couplage entre Modules

### 5.1 Dépendances Correctes

```
dnd5e.mjs
  ├── config.mjs (configuration pure)
  ├── utils.mjs (utilitaires purs)
  ├── applications/ (UI)
  │     └── dépend de documents, data
  ├── documents/ (logique métier)
  │     └── dépend de data, config
  └── data/ (modèles de données)
        └── dépend de config
```

### 5.2 Couplages Problématiques

1. **enrichers.mjs ↔ documents/actor/trait.mjs** : Couplage fort pour les lookups
2. **config.mjs ↔ data/activity/** : Import circulaire potentiel via `ConsumptionTargetData`
3. **registry.mjs ↔ applications/compendium-browser.mjs** : Dépendance UI dans le registry

---

## 6. Recommandations pour un Nouveau Système v13

### 6.1 À Reproduire

1. **Structure modulaire** avec barrels (`_module.mjs`)
2. **Configuration centralisée** mais découpée par domaine
3. **Système de pré-localisation**
4. **Hooks bien organisés** (init → setup → i18nInit → ready)
5. **Enrichers de texte** pour les descriptions interactives
6. **Migrations versionnées**

### 6.2 À Améliorer

1. **Utiliser TypeScript** ou au minimum des types stricts
2. **Découper config.mjs** en fichiers par domaine
3. **Éviter les IDs hardcodés** pour les compendiums
4. **Ajouter des tests unitaires**
5. **Simplifier le système d'activités** si possible
6. **Documenter l'API publique**

### 6.3 À Éviter

1. Fichiers de 2000+ lignes
2. Gestion simultanée de versions de règles incompatibles
3. Couplage fort avec les données de compendium
4. Fonctions utilitaires "fourre-tout"
5. Absence de validation de types

---

## 7. Conclusion

Le système dnd5e est une **excellente référence architecturale** pour la structure modulaire et l'organisation du code. Cependant, son historique long a créé une dette technique significative.

Pour un nouveau système FoundryVTT v13, il faut :
- **S'inspirer** de l'architecture globale et des patterns de configuration
- **Moderniser** avec TypeScript et des tests
- **Simplifier** la complexité héritée
- **Découper** les fichiers monolithiques

Le ratio effort/bénéfice est favorable si on extrait les bonnes pratiques tout en évitant les erreurs accumulées au fil des versions.
